package cn.com.duiba.wolf.cache;

import cn.com.duiba.boot.ext.autoconfigure.redis.Hessian2SerializationRedisSerializer;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.entity.Null;
import cn.com.duiba.wolf.redis.RedisClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.TimeoutUtils;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.io.Serializable;
import java.util.concurrent.TimeUnit;

/**
 * Created by wenqi.huang on 2017/1/11.
 */
public class RedisCacheClient implements AdvancedCacheClient {

    private static final Logger logger = LoggerFactory.getLogger(RedisCacheClient.class);

    private final RedisTemplate redisTemplate;

    private RedisSerializer<String> keyRedisSerializer = new StringRedisSerializer();
    private RedisSerializer<Object> valueRedisSerializer = new Hessian2SerializationRedisSerializer();

    public RedisCacheClient(RedisTemplate redisTemplate){
        this.redisTemplate = redisTemplate;
    }

    @Override
    public <T> T get(String key) {
        T t = getInner(key);
        if(t != null && t instanceof Null){//如果取回的数据是NullCache,表示是之前故意放入的,特殊处理,返回null
            return null;
        }
        return t;
    }

    @Override
    public boolean set(final String key, final Object value, final int timeout, final TimeUnit unit) {
        if(value == null){
            return false;
        }

        final byte[] rawKey = rawKey(key);
        final byte[] rawValue = rawValue(value);

        try {
            redisTemplate.execute(new RedisCallback<Object>() {

                public Object doInRedis(RedisConnection connection) throws DataAccessException {

                    potentiallyUsePsetEx(connection);
                    return null;
                }

                public void potentiallyUsePsetEx(RedisConnection connection) {

                    if (!TimeUnit.MILLISECONDS.equals(unit) || !failsafeInvokePsetEx(connection)) {
                        connection.setEx(rawKey, TimeoutUtils.toSeconds(timeout, unit), rawValue);
                    }
                }

                private boolean failsafeInvokePsetEx(RedisConnection connection) {

                    boolean failed = false;
                    try {
                        connection.pSetEx(rawKey, timeout, rawValue);
                    } catch (UnsupportedOperationException e) {
                        // in case the connection does not support pSetEx return false to allow fallback to other operation.
                        failed = true;
                    }
                    return !failed;
                }

            }, true);
        } catch(Exception e){
            logger.error(e.getMessage(), e);
            return false;
        }

        return true;
    }

    private <T> T getInner(String key) {
        final byte[] rawKey = rawKey(key);
        try {
            return (T)redisTemplate.execute(new RedisCallback<T>() {
                @Override
                public T doInRedis(RedisConnection connection) throws DataAccessException {
                    byte[] valueBytes = connection.get(rawKey);
                    return (T) valueRedisSerializer.deserialize(valueBytes);
                }
            }, true);
        } catch(Exception e){
            logger.error("get key error:"+key, e);
            return null;
        }
    }

    @Override
    public <T> T getWithCacheLoader(String key, int timeout, TimeUnit timeUnit, boolean isCacheNull, CacheLoader<T> cacheLoader) {
        T value = getInner(key);
        if(value == null){
            value = cacheLoader.load();
            if(isCacheNull) {
                setWithNull(key, value, timeout, timeUnit);
            }else if(value != null){
                set(key, value, timeout, timeUnit);
            }
        }

        if(value instanceof Null){//如果取回的数据是NullCache,表示是之前故意放入的,特殊处理,返回null
            value = null;
        }

        return value;
    }

    protected void setWithNull(String key, Object value, int timeout, TimeUnit unit) {
        if (value == null) {
            value = Null.NULL;
        }
        set(key, value, timeout, unit);
    }

    @Override
    public <T> T getWithCacheLoader(String key, int exp, TimeUnit timeUnit, CacheLoader<T> cacheLoader) {
        return getWithCacheLoader(key, exp, timeUnit, false, cacheLoader);
    }

    @Override
    public boolean remove(String key) {
        final byte[] rawKey = rawKey(key);
        try {
            redisTemplate.execute(new RedisCallback<Object>() {

                @Override
                public Object doInRedis(RedisConnection connection) throws DataAccessException {
                    connection.del(rawKey);
                    return null;
                }
            }, true);
        } catch(Exception e){
            logger.error(e.getMessage(), e);
            return false;
        }
        return true;
    }

    private byte[] rawKey(String key){
        return keyRedisSerializer.serialize(key);
    }

    private byte[] rawValue(Object value){
        return valueRedisSerializer.serialize(value);
    }

    public static void main(String[] args) throws InterruptedException {
        JedisConnectionFactory factory = new JedisConnectionFactory();
        factory.setHostName("dev.config.duibar.com");
        factory.setPort(6379);
        factory.setPassword("duiba123");
        factory.setTimeout(100);
        factory.afterPropertiesSet();
        RedisTemplate stringRedisTemplate = new RedisTemplate();
        stringRedisTemplate.setConnectionFactory(factory);
        stringRedisTemplate.afterPropertiesSet();

        RedisCacheClient redisCacheClient = new RedisCacheClient(stringRedisTemplate);

        RedisClient redisClient = new RedisClient(false,"master","dev.config.duibar.com:6379","duiba123");

        redisCacheClient.set("name","hwq", 1000, TimeUnit.MILLISECONDS);
        String value = redisCacheClient.get("name");
        System.out.println(value);

        redisCacheClient.set("name","100", 10000000, TimeUnit.MILLISECONDS);
        System.out.println(redisClient.get("name"));

        DubboResult d = DubboResult.failResult("???jjjj");
        redisCacheClient.set("name",d, 10000000, TimeUnit.MILLISECONDS);
        d = redisCacheClient.get("name");
        System.out.println(d.getMsg());

        RedisTemplate<String, DubboResult> redisTemplate = new RedisTemplate<String, DubboResult>();
        redisTemplate.setEnableDefaultSerializer(false);
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.setValueSerializer(new Hessian2SerializationRedisSerializer());
        redisTemplate.setConnectionFactory(factory);
        redisTemplate.afterPropertiesSet();

        d = redisTemplate.opsForValue().get("name");
        System.out.println(d.getMsg());

    }
}
