package cn.com.duiba.wolf.dubbo;

import java.io.PrintWriter;
import java.lang.reflect.Method;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.alibaba.dubbo.common.Constants;
import com.alibaba.dubbo.common.extension.Activate;
import com.alibaba.dubbo.common.io.UnsafeStringWriter;
import com.alibaba.dubbo.rpc.*;
import com.alibaba.dubbo.rpc.service.GenericService;
import org.springframework.core.NestedRuntimeException;

/**
 * Created by wenqi.huang on 16/6/1.<br/>
 * dubbo拦截器,写法参考dubbo自带的ExceptionFilter
 */
@Activate(group = { Constants.PROVIDER }, order = -8999)
public class RuntimeExceptionFilter implements Filter {

    private static final Logger log = LoggerFactory.getLogger(RuntimeExceptionFilter.class);

    /**
     * 系统名,两位,比如DS表示duibaservice、GC表示goods-center等。详见http://cf.dui88.com/pages/viewpage.action?pageId=3544570
     */
    private static String SYSTEM_NAME = "--";

    /**
     * 表示可以定义为服务器内部异常的类,有待继续收集
     */
    private static final Set<Class<? extends RuntimeException>> serviceInternalRuntimeExceptions = new HashSet<>();

    static{
        serviceInternalRuntimeExceptions.add(NestedRuntimeException.class);
    }

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        try {
            DBTimeProfile.enter("RuntimeExceptionFilter start");
            Result result = invoker.invoke(invocation);
            if (result.hasException() && GenericService.class != invoker.getInterface()) {
                try {
                    Throwable exception = result.getException();

                    // 如果是checked异常，直接抛出
                    if (!(exception instanceof RuntimeException) && (exception instanceof Exception)) {
                        return result;
                    }

                    // 是Dubbo本身的异常，直接抛出
                    if (exception instanceof RpcException) {
                        return result;
                    }

                    try {
                        // 在方法签名上有声明，直接抛出
                        Method method = invoker.getInterface().getMethod(invocation.getMethodName(),
                                                                         invocation.getParameterTypes());
                        Class<?>[] exceptionClassses = method.getExceptionTypes();
                        for (Class<?> exceptionClass : exceptionClassses) {
                            if (exception.getClass().equals(exceptionClass)) {
                                return result;
                            }
                        }

                        Class<?> clazz = method.getReturnType();
                        if (clazz.equals(Void.TYPE)) {
                            return result;// 接口返回类型为void则强制抛出异常给客户端
                        } else if (clazz.equals(DubboResult.class)) {// 只有当返回类型为DubboResult时才应该捕获RuntimeException并封装错误信息
                            logException(invocation.getArguments(), exception);

                            String exceptionMsg = exception.getMessage();

                            //如果是RuntimeException（但不是子类）,且子异常为空,而且消息非空,则认为是应用级错误,很多地方这么抛出了,以后不应该抛出这样的异常
                            if(exception.getCause() == null && exception.getClass().equals(RuntimeException.class) && !StringUtils.isEmpty(exceptionMsg)){
                                return new RpcResult(DubboResult.failResult(exception.getMessage()));
                            }
                            //处理服务器内部异常,大部分RuntimeException子类都可以认为是服务器内部异常
                            else if(serviceInternalRuntimeExceptions.contains(exception.getClass())){
                                DubboResult dubboResult = DubboResult.failResult("服务器内部错误");
                                dubboResult.setReturnCode(SYSTEM_NAME + "0102");
                                return new RpcResult(dubboResult);
                            }

                            return new RpcResult(DubboResult.failResult(exception.getMessage()));
                        } else {
                            return result;// 强制抛出异常给客户端
                        }
                    } catch (NoSuchMethodException e) {
                        return result;
                    }

                    // 包装成RuntimeException抛给客户端（把异常堆栈缩小,每个causeBy只保留两行）
                    //return new RpcResult(new DubboException(exception.getMessage(), toShortString(exception)));
                    //StringUtils.toString(exception);
                } catch (Exception e) {
                    return result;
                }
            }
            return result;
        } catch (RuntimeException e) {
            throw e;
        } finally {
            DBTimeProfile.release();
        }
    }

    private void logException(Object[] arguments, Throwable exception){
        StringBuilder message = new StringBuilder(exception.getMessage());
        message.append(" params : ");
        String argumentsStr = Arrays.toString(arguments);
        int maxArgumentsStrLen = 300;
        if(argumentsStr != null && argumentsStr.length() > maxArgumentsStrLen){//参数字符串过长进行截取
            argumentsStr = argumentsStr.substring(0, maxArgumentsStrLen);
            message.append(argumentsStr).append(" ... (").append(argumentsStr.length() - maxArgumentsStrLen).append(" more characters).");
        }else{
            message.append(argumentsStr);
        }

        //打印异常日志,方便排查问题
        log.error(message.toString(), exception);
    }

    /**
     *
     * @param e
     * @return string
     */
    private String toShortString(Throwable e) {
        UnsafeStringWriter w = new UnsafeStringWriter();
        PrintWriter p = new PrintWriter(w);
        try {
            //e.printStackTrace(p);

            // Print our stack trace
            printStackTrace(e, p, true);
//            // Print suppressed exceptions, if any
//            for (Throwable se : e.getSuppressed())
//                se.printEnclosedStackTrace(p, trace, "Suppressed: ", "\t", dejaVu);

            // Print cause, if any
//            Throwable ourCause = e.getCause();
//            if (ourCause != null)
//                ourCause.printEnclosedStackTrace(p, trace, "Caused by: ", "", dejaVu);

            return w.toString();
        } finally {
            p.close();
        }
    }

    private void printStackTrace(Throwable e, PrintWriter p, boolean isTop){
        if(e != null){
            if(isTop) {
                p.println(e);
            }else{
                p.println("Caused by: " + e);
            }
            StackTraceElement[] trace = e.getStackTrace();
            int i = 0;
            int lastPrintedLineNumber = 0;
            for (StackTraceElement traceElement : trace) {
                if(i < 2 || traceElement.getClassName().startsWith("cn.com.duiba")) {
                    if(lastPrintedLineNumber < i - 1){
                        p.println("\t...");
                    }
                    p.println("\tat " + traceElement);
                    lastPrintedLineNumber = i;
                }
                i++;
            }

            printStackTrace(e.getCause(), p, false);
        }
    }

    /**
     * 注册系统名字, 两位,比如DS表示duibaservice、GC表示goods-center等。详见http://cf.dui88.com/pages/viewpage.action?pageId=3544570
     * <br/><br/>
     * @param systemName
     */
    protected static void registerSystemName(String systemName) {
        if(StringUtils.isEmpty(systemName)) {
            throw new IllegalArgumentException("systemName must not be null");
        } else if(systemName.length() != 2){
            throw new IllegalArgumentException("systemName's length must be 2");
        }

        SYSTEM_NAME = systemName;
    }

}
