package cn.com.duiba.wolf.perf.timeprofile;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

public class DBTimeProfile {

    private static final Logger                               log       = LoggerFactory.getLogger(DBTimeProfile.class);
    private static final ThreadLocal<LinkedList<TimeProfileElement>> stack     = new ThreadLocal<>();
    private static final ThreadLocal<LinkedList<TimeProfileElement>>  logs      = new ThreadLocal<>();
    private static final ThreadLocal<HttpServletRequest>              request   = new ThreadLocal<>();
    private static int                                  threshold = 100;// 超时时间，单位毫秒
    private static final String timeout = "DBTimeProfile timeout ";

    private DBTimeProfile(){}

    private static void enter() {
        enter("");
    }

    public static void enter(String tag) {
        if (stack.get() == null) {
            return;
        }
        TimeProfileElement t = new TimeProfileElement();
        t.logEnterTime();
        t.setTag(tag);
        t.setDeep(stack.get().size());

        stack.get().push(t);
    }

    public static void release() {
        LinkedList<TimeProfileElement> s1 = stack.get();
        if (s1 == null) {
            return;
        }
        if(s1.size() <= 1){
            //throw new IllegalStateException("release failed,will stop this timeProfile, enter/release 必须成对出现");
            log.warn("release failed,will stop this timeProfile, enter/release 必须成对出现");
            clear();
            return;
        }
        TimeProfileElement t = stack.get().pop();
        t.logReleaseTime();
        logs.get().add(t);
    }

    public static void start() {
        stack.set(new LinkedList<TimeProfileElement>());
        logs.set(new LinkedList<TimeProfileElement>());
        enter();
    }

    public static void end() {
        end("undefined");
    }

    public static void end(String methodName) {
        LinkedList<TimeProfileElement> s1 = stack.get();
        if(s1 == null){
            return;
        }
        if(s1.size() != 1){
            log.warn("timeProfile methodName:{} failed, enter/release 必须成对出现, stackSize:{}", methodName, s1.size());
            clear();
            return;
        }
        TimeProfileElement t = s1.pop();
        t.logReleaseTime();
        logs.get().add(t);

        long timeConsume = t.getCostTimeMillis();
        if (t.getCostTimeMillis() > threshold) {
            // 输出日志
            StringBuilder sb = new StringBuilder();
            if (request.get() != null) {
                String url = RequestTool.getRequestUrl(request.get());
                sb.append(timeout).append(timeConsume).append("ms >").append(threshold).append("ms, url=").append(url);
            } else {
                sb.append(timeout).append(timeConsume).append("ms > methodName:").append(methodName);
            }
            List<TimeProfileElement> list = new ArrayList<>(logs.get());


            Collections.reverse(list);
            //listSort(list);
            for (TimeProfileElement s : list) {
                sb.append("\r\n\t");
                for (int i = 0; i < s.getDeep(); i++) {
                    sb.append("-");
                }
                Long consume = s.getCostTimeMillis();
                sb.append(consume * 100 / timeConsume).append("%");
                sb.append("  ").append(consume).append("ms");
                if (s.getTag() != null) {
                    sb.append("  ").append(s.getTag());
                }
            }
            log.warn("{}",sb);
        }
        clear();
    }

    private static void clear(){
        request.set(null);
        stack.set(null);
        logs.set(null);
    }

    public static void setThreshold(int threshold) {
        DBTimeProfile.threshold = threshold;
    }

    /**
     * dump堆栈信息
     *
     */
    private static String dump() {
        if (logs.get() == null) {
            return null;
        }
        TimeProfileElement t = stack.get().pop();
        t.logReleaseTime();
        logs.get().add(t);

        Long timeconsume = 0L;
        for (TimeProfileElement s : logs.get()) {
            if (s.deep == 0) {
                timeconsume += s.getCostTimeMillis();
            }
        }
        if (timeconsume < (long) 0) {
            return null;
        }
        // 输出日志
        StringBuilder sb = new StringBuilder();
        sb.append(timeout).append(timeconsume).append("ms ");
        List<TimeProfileElement> list = new ArrayList<>(logs.get());
        Collections.reverse(list);
        for (TimeProfileElement s : list) {
            Long consume = s.getCostTimeMillis();
            if (consume == 0) {
                continue;
            }
            sb.append("\r\n\t");
            for (int i = 0; i < s.getDeep(); i++) {
                sb.append("  ");
            }
            sb.append("-");
            sb.append(consume * 100 / timeconsume).append("%");
            sb.append("  ").append(consume).append("ms");
            if (s.getTag() != null) {
                sb.append("  ").append(s.getTag());
            }
        }
        clear();

        return sb.toString();
    }

    public static void main(String[] args) throws Exception {
        DBTimeProfile.start();
        try {
            try {
                DBTimeProfile.enter("in");
                Thread.sleep(1000);
                throw new NullPointerException("");
            } finally {
                DBTimeProfile.release();
            }
        }finally {
            DBTimeProfile.end();
            //System.out.println(DBTimeProfile.dump());
        }
    }

    public static class TimeProfileElement {

        //基线时间，为了减小timeMillis的占用空间
        private static final long BASE_TIME = System.currentTimeMillis();

        private int timeMillis;
        private int               deep;
        private String            tag;

        void logEnterTime(){
            timeMillis = Long.valueOf(System.currentTimeMillis() - BASE_TIME).intValue();
        }

        void logReleaseTime(){
            timeMillis = Long.valueOf(System.currentTimeMillis() - BASE_TIME - timeMillis).intValue();
        }

        long getCostTimeMillis(){
            return timeMillis;
        }

        int getDeep() {
            return deep;
        }

        void setDeep(int deep) {
            this.deep = deep;
        }

        String getTag() {
            return tag;
        }

        void setTag(String tag) {
            this.tag = tag;
        }

    }
}
