package cn.com.duiba.wolf.log;

import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import org.slf4j.Logger;
import org.slf4j.Marker;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 自动降级的logger代理,只支持logback，这个类的作用是防止日志打印qps高时导致log阻塞。该日志代理有两个阈值：
 * <br/>
 * 如果打印日志的qps超过了第一个阈值（默认为5），则忽略该日志的错误堆栈，而只打印错误消息；
 * <br/>
 * 如果qps超过了第二个阈值（默认为10），则直接忽略此消息（不打印）
 * <br/>
 * 如果qps低则正常打印日志，
 * Created by wenqi.huang on 2017/2/10.
 */
public class DegradeLogger implements Logger {

    private static final LoadingCache<Logger, AtomicInteger> logger2QpsCache = Caffeine.newBuilder()
            .expireAfterWrite(1, TimeUnit.SECONDS)
            .initialCapacity(10)
            .maximumSize(1000000)//最多存100W个logger
            .build(key -> new AtomicInteger(0));

    private final Logger logger;

    /**
     * 日志qps高的水位线，如果该日志qps超过此阈值，则对于INFO和WARN日志，忽略该日志的错误堆栈，而只打印错误消息；对于INFO、DEBUG日志，直接忽略
     */
    private int waterLevelHign;
    /**
     * 日志qps非常高的水位线，如果该日志qps超过此阈值，则在当前秒内对应的日志不再打印
     */
    private int waterLevelVeryHigh;

    private DegradeLogger(Logger logger, int waterLevelHign, int waterLevelVeryHigh) {
        this.logger = logger;
        this.waterLevelHign = waterLevelHign;
        this.waterLevelVeryHigh = waterLevelVeryHigh;
    }

    public static DegradeLogger wrap(Logger logger, int waterLevelHign, int waterLevelVeryHigh){
        if(logger == null){
            return null;
        }
        if(logger instanceof DegradeLogger){
            return (DegradeLogger)logger;
        }
        return new DegradeLogger(logger, waterLevelHign, waterLevelVeryHigh);
    }

    public static DegradeLogger wrap(Logger logger, int waterLevelHign){
        return wrap(logger, waterLevelHign, 10);
    }

    public static DegradeLogger wrap(Logger logger){
        return wrap(logger, 5);
    }

    @Override
    public String getName() {
        return logger.getName();
    }

    @Override
    public boolean isTraceEnabled() {
        return logger.isTraceEnabled();
    }

    @Override
    public void trace(String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(msg);
    }

    @Override
    public void trace(String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(format, arg);
    }

    @Override
    public void trace(String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(format, arg1, arg2);
    }

    @Override
    public void trace(String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(format, arguments);
    }

    @Override
    public void trace(String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.trace("{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.trace(msg, t);
        }
    }

    @Override
    public boolean isTraceEnabled(Marker marker) {
        return logger.isTraceEnabled(marker);
    }

    @Override
    public void trace(Marker marker, String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(marker, msg);
    }

    @Override
    public void trace(Marker marker, String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(marker, format, arg);
    }

    @Override
    public void trace(Marker marker, String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(marker, format, arg1, arg2);
    }

    @Override
    public void trace(Marker marker, String format, Object... argArray) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.trace(marker, format, argArray);
    }

    @Override
    public void trace(Marker marker, String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.trace(marker, "{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.trace(marker, msg, t);
        }
    }

    @Override
    public boolean isDebugEnabled() {
        return logger.isDebugEnabled();
    }

    @Override
    public void debug(String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(msg);
    }

    @Override
    public void debug(String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(format, arg);
    }

    @Override
    public void debug(String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(format, arg1, arg2);
    }

    @Override
    public void debug(String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(format, arguments);
    }

    @Override
    public void debug(String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.debug("{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.debug(msg, t);
        }
    }

    @Override
    public boolean isDebugEnabled(Marker marker) {
        return logger.isDebugEnabled(marker);
    }

    @Override
    public void debug(Marker marker, String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(marker, msg);
    }

    @Override
    public void debug(Marker marker, String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(marker, format, arg);
    }

    @Override
    public void debug(Marker marker, String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(marker, format, arg1, arg2);
    }

    @Override
    public void debug(Marker marker, String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.debug(marker, format, arguments);
    }

    @Override
    public void debug(Marker marker, String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.debug(marker, "{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.debug(marker, msg, t);
        }
    }

    @Override
    public boolean isInfoEnabled() {
        return logger.isInfoEnabled();
    }

    @Override
    public void info(String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(msg);
    }

    @Override
    public void info(String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(format, arg);
    }

    @Override
    public void info(String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(format, arg1, arg2);
    }

    @Override
    public void info(String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(format, arguments);
    }

    @Override
    public void info(String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.info("{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.info(msg, t);
        }
    }

    @Override
    public boolean isInfoEnabled(Marker marker) {
        return logger.isInfoEnabled(marker);
    }

    @Override
    public void info(Marker marker, String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(marker, msg);
    }

    @Override
    public void info(Marker marker, String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(marker, format, arg);
    }

    @Override
    public void info(Marker marker, String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(marker, format, arg1, arg2);
    }

    @Override
    public void info(Marker marker, String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.info(marker, format, arguments);
    }

    @Override
    public void info(Marker marker, String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.info(marker, "{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.info(marker, msg, t);
        }
    }

    @Override
    public boolean isWarnEnabled() {
        return logger.isWarnEnabled();
    }

    @Override
    public void warn(String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(msg);
    }

    @Override
    public void warn(String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(format, arg);
    }

    @Override
    public void warn(String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(format, arguments);
    }

    @Override
    public void warn(String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(format, arg1, arg2);
    }

    @Override
    public void warn(String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.warn("{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.warn(msg, t);
        }
    }

    @Override
    public boolean isWarnEnabled(Marker marker) {
        return logger.isWarnEnabled(marker);
    }

    @Override
    public void warn(Marker marker, String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(marker, msg);
    }

    @Override
    public void warn(Marker marker, String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(marker, format, arg);
    }

    @Override
    public void warn(Marker marker, String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(marker, format, arg1, arg2);
    }

    @Override
    public void warn(Marker marker, String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.warn(marker, format, arguments);
    }

    @Override
    public void warn(Marker marker, String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.warn(marker, "{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.warn(marker, msg, t);
        }
    }

    @Override
    public boolean isErrorEnabled() {
        return logger.isErrorEnabled();
    }

    @Override
    public void error(String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.error(msg);
    }

    @Override
    public void error(String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.error(format, arg);
    }

    @Override
    public void error(String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.error(format, arg1, arg2);
    }

    @Override
    public void error(String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.error(format, arguments);
    }

    @Override
    public void error(String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.error("{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.error(msg, t);
        }
    }

    @Override
    public boolean isErrorEnabled(Marker marker) {
        return logger.isErrorEnabled(marker);
    }

    @Override
    public void error(Marker marker, String msg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        logger.error(marker, msg);
    }

    @Override
    public void error(Marker marker, String format, Object arg) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }

        logger.error(marker, format, arg);
    }

    @Override
    public void error(Marker marker, String format, Object arg1, Object arg2) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }

        logger.error(marker, format, arg1, arg2);
    }

    @Override
    public void error(Marker marker, String format, Object... arguments) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }

        logger.error(marker, format, arguments);
    }

    @Override
    public void error(Marker marker, String msg, Throwable t) {
        int qps = incrAndGetLogQps();
        if(isQpsVeryHigh(qps)){
            return;
        }
        else if(isQpsHigh(qps)){
            logger.error(marker, "{},errorMessage:{}", msg, t.getMessage());
        }else{
            logger.error(marker, msg, t);
        }
    }

    /**
     * 增加log的qps，并返回增加后的qps值
     * @return
     */
    private int incrAndGetLogQps(){
        try {
            AtomicInteger qps = logger2QpsCache.get(logger);
            return qps.incrementAndGet();
        } catch (Exception e) {
            logger.warn("", e);
            //Ignore
            return 0;
        }
    }

    /**
     * 判断当前logger的qps是否高
     * @return
     */
    private boolean isQpsHigh(int qps){
        return qps > waterLevelHign;
    }

    /**
     * 判断当前logger的qps是否太高
     * @return
     */
    private boolean isQpsVeryHigh(int qps){
        return qps > waterLevelVeryHigh;
    }

}
