package cn.com.duiba.wolf.utils;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.Hex;

import java.security.SecureRandom;
import java.util.UUID;

/**
 * Created by wenqi.huang on 16/6/16.
 */
public class UUIDUtils {
    /**
     * 生成一个32位uuid，注意此方法生成的uuid是不安全的uuid，顺序性较强，随机性弱，不能用于sessionId等场合。
     * @return
     */
    public static String createUUID(){
        return UUID.randomUUID().toString().replaceAll("-", "");
    }

    /**
     * 生成一个22位的更短的uuid（本质上内部是base64编码了下），可以拿来做随机图片链接
     * @return
     */
    public static String createCompressedUUID() {
        UUID uuid = UUID.randomUUID();
        return compressedUUID(uuid);
    }

    /**
     * 生成一个32位安全uuid，可以用于sessionId、作为密钥等场合。
     * @return
     */
    public static String createSecureUUID(){
        SecureRandom ranGen = new SecureRandom();
        byte[] bs = new byte[16]; // 16 bytes = 128 bits
        ranGen.nextBytes(bs);
        return Hex.encodeHexString(bs);
    }

    /**
     * 生成一个22位更短的安全uuid，可以用于sessionId、作为密钥等场合。
     * @return
     */
    public static String createCompressedSecureUUID(){
        SecureRandom ranGen = new SecureRandom();
        byte[] bs = new byte[16]; // 16 bytes = 128 bits
        ranGen.nextBytes(bs);
        return Base64.encodeBase64URLSafeString(bs);
    }

    private static String compressedUUID(UUID uuid) {
        byte[] byUuid = new byte[16];
        long least = uuid.getLeastSignificantBits();
        long most = uuid.getMostSignificantBits();
        long2bytes(most, byUuid, 0);
        long2bytes(least, byUuid, 8);
        return Base64.encodeBase64URLSafeString(byUuid);
    }

    private static void long2bytes(long value, byte[] bytes, int offset) {
        for (int i = 7; i > -1; i--) {
            bytes[offset++] = (byte) ((value >> 8 * i) & 0xFF);
        }
    }

    /**
     * 把压缩过的uuid还原位32位
     * @param compressedUuid
     * @return
     */
    public static String uncompress(String compressedUuid) {
        if (compressedUuid.length() != 22) {
            throw new IllegalArgumentException("Invalid uuid!");
        }
        byte[] byUuid = Base64.decodeBase64(compressedUuid + "==");
        long most = bytes2long(byUuid, 0);
        long least = bytes2long(byUuid, 8);
        UUID uuid = new UUID(most, least);
        return uuid.toString().replaceAll("-", "");
    }

    private static long bytes2long(byte[] bytes, int offset) {
        long value = 0;
        for (int i = 7; i > -1; i--) {
            value |= (((long) bytes[offset++]) & 0xFF) << 8 * i;
        }
        return value;
    }

}
