/**
 * Copyright (c) 2019, duiba.com.cn All Rights Reserved.
 */
package cn.tuia.tools.deviceid.v2;

import org.apache.commons.lang.StringUtils;

import java.util.regex.Pattern;

/**
 * 描述: imei & idfa 公共方法
 *
 * @version v1.0
 * @auther guopengfei@duiba.com.cn
 * @date 2019/7/4 17:06
 */
public class ImeiAndIdfaUtils {

    private static Pattern PATTERN_IMEI_ONE = Pattern.compile("\\d{15}$");

    private static Pattern PATTERN_IMEI_TWO = Pattern.compile("^(A|a)\\d{14}$");

    private static Pattern PATTERN_IDFA = Pattern.compile("^([0-9A-Z]{1,})(([/\\s-][0-9A-Z]{1,}){4})$");

    private static Pattern PATTERN_IDFA_LENGTH = Pattern.compile("^.{36}$");

    private static Pattern PATTERN_MD5_LOWER_CASE = Pattern.compile("^([0-9a-f]{32})$");

    private static Pattern PATTERN_MD5_UPPER_CASE = Pattern.compile("^([0-9A-F]{32})$");

    /**
     * 入参是否真实imei
     * <p>
     * 1、长度是否等于15
     * 2、是否纯数字 || 是否A+14位纯数字
     *
     * @param imei
     * @return
     */
    public static final boolean isImei(String imei) {
        if (StringUtils.isNotBlank(imei)) {
            return PATTERN_IMEI_ONE.matcher(imei.trim()).matches() ||
                    PATTERN_IMEI_TWO.matcher(imei.trim()).matches();
        }
        return false;
    }

    /**
     * 入参是否真实idfa
     * <p>
     * 1、直接正则匹配
     *
     * @param idfa
     * @return
     */
    public static final boolean isIdfa(String idfa) {
        if (StringUtils.isNotBlank(idfa)) {
            return PATTERN_IDFA.matcher(idfa).matches() && PATTERN_IDFA_LENGTH.matcher(idfa).matches();
        }
        return false;
    }

    /**
     * 入参是否真实oaid
     * <p>
     * oaid仅做一下简单的校验规则，防止通用广告联盟oaid的宏没有替换成功
     * 宏的替换内容一般是 __100000001__  这样的格式
     * 以后有具体的规则，你可以继续维护
     *
     * @param oaid
     * @return
     */
    public static final boolean isOaid(String oaid) {
        if (StringUtils.isBlank(oaid)) {
            return false;
        }
        if (oaid.startsWith("__") || oaid.endsWith("__")) {
            return false;
        }
        return true;
    }

    /**
     * 是否小写32位MD5加密字符串
     *
     * @param str
     * @return
     */
    public static final boolean isLowerCase32Md5(String str) {
        if (StringUtils.isNotBlank(str)) {
            return PATTERN_MD5_LOWER_CASE.matcher(str.trim()).matches();
        }
        return false;
    }

    /**
     * 是否大写32位MD5加密字符串
     *
     * @param str
     * @return
     */
    public static final boolean isUpperCase32Md5(String str) {
        if (StringUtils.isNotBlank(str)) {
            return PATTERN_MD5_UPPER_CASE.matcher(str.trim()).matches();
        }
        return false;
    }

    // -- 黑名单

    /**
     * 是否黑名单设备号
     *
     * @param deviceId
     * @return
     */
    public static final boolean isBlackDevice(String deviceId) {
        if (StringUtils.isBlank(deviceId)) {
            return false;
        }
        if (deviceId.startsWith("__") && deviceId.endsWith("__")) {
            return true;
        }
        return false;
    }

}
