package cn.tuia.tools.ua;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;

import java.net.URLDecoder;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 描述: UserAgent信息清洗工具
 *
 * @author guopengfei@duiba.com.cn
 * @version v1.0
 * @date 2020/11/13 14:21
 */
@Slf4j
public class UACleanUtils {

    /**
     * 清洗 userAgent
     *
     * @param userAgent
     * @return
     */
    public static String clean(String userAgent) {
        if (StringUtils.isBlank(userAgent)) {
            return StringUtils.EMPTY;
        }
        userAgent = handleUserAgent(userAgent);

        if (!userAgent.contains("Android")) {
            return handleUserAgentDefault(userAgent);
        }

        String result = handleUserAgentAndroid(userAgent);
        if (StringUtils.isBlank(result)) {
            return handleUserAgentDefault(userAgent);
        }
        return result;
    }

    /**
     * 参数预处理（去空格和UrlDecode）
     *
     * @param userAgent
     * @return 处理后的参数
     */
    private static String handleUserAgent(String userAgent) {
        try {
            userAgent = URLDecoder.decode(userAgent.trim(), "utf-8");
        } catch (Exception ignored) {
            // do nothing
        }

        return userAgent;
    }

    /**
     * 处理UserAgent的默认策略
     *
     * @param ua UserAgent
     * @return 处理后的UserAgent
     */
    private static String handleUserAgentDefault(String ua) {
        Pattern pattern = Pattern.compile("\\(.*?\\)");
        Matcher matcher = pattern.matcher(ua);
        if (matcher.find()) {
            ua = StringUtils.defaultString(matcher.group());
        }
        ua = ua.replaceAll(" U;", "");
        ua = ua.replaceAll("; wv", "");
        ua = ua.replaceAll("\\s\\w\\w-\\w\\w;", "");
        ua = ua.replaceAll("0.0;", "0;");
        ua = ua.replaceAll("1.0;", "1;");
        ua = ua.replaceAll("zh-cn;", "");
        ua = ua.replaceAll("zh-CN;", "");
        ua = ua.replaceAll(" ", "");
        ua = ua.replaceAll("%20", "");
        return ua;
    }

    /**
     * 处理UserAgent
     *
     * @param ua UserAgent
     * @return 处理后的UserAgent
     */
    private static String handleUserAgentAndroid(String ua) {
        ua = ua.replaceAll(" U;", "");
        ua = ua.replaceAll("; wv", "");
        ua = ua.replaceAll("\\s\\w\\w-\\w\\w;", "");

        Matcher matcher = Pattern.compile("Android\\s\\d.*?/").matcher(ua);
        if (!matcher.find()) {
            return "";
        }

        ua = matcher.group();
        String uas[] = ua.split(";");
        if (uas.length < 1) {
            return "";
        }
        String system = "";
        String model = "";
        for (String uaStr : uas) {
            if (uaStr.contains("Android")) {
                Matcher systemMatcher = Pattern.compile("Android\\s\\d+").matcher(uaStr);
                if (systemMatcher.find()) {
                    system = systemMatcher.group();
                }
            } else if (uaStr.contains("/")) {
                model = uaStr.trim().split(" ")[0];
            }
        }

        return StringUtils.isNotBlank(system) && StringUtils.isNotBlank(model) ?
                (system + model).replaceAll(" ", "") :
                StringUtils.EMPTY;
    }

}
