package cn.com.duiba.activity.center.api.enums;


import cn.com.duiba.activity.center.api.dto.sign.SignRuleConfigDto;
import cn.com.duiba.activity.center.api.dto.sign.SignRuleItemDto;
import cn.com.duiba.activity.center.api.exception.ActivityCenterException;
import cn.com.duiba.activity.center.api.exception.ActivityCenterRuntimeException;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.DateUtils;

import java.util.*;

/**
 * 签到类型
 * Created by xiaoxuda on 2017/5/31.
 */
public enum SignTypeEnum {
    CONTINUE(0, "连续签到") {
        @Override
        public SignRuleItemDto getRuleItemSub(int continueDay, int dayFeature, SignRuleConfigDto signRuleConfig) {
            continueDay += dayFeature;
            int circle = signRuleConfig.getItems().size();
            int index = continueDay > circle ? circle : continueDay;
            return signRuleConfig.getItems().get(index - 1);
        }
    },
    CIRCLE(1, "周期签到") {
        @Override
        public SignRuleItemDto getRuleItemSub(int continueDay, int dayFeature, SignRuleConfigDto signRuleConfig) {
            continueDay += dayFeature;
            int circle = signRuleConfig.getItems().size();
            int index = continueDay % circle;
            index = index == 0 ? circle : index;
            return signRuleConfig.getItems().get(index - 1);
        }
    },
    CUSTOM(2, "自定义签到") {
        @Override
        public SignRuleItemDto getRuleItemSub(int continueDay, int dayFeature, SignRuleConfigDto signRuleConfig) {
            return null;
        }
    },
    MONTH(3, "自然月签到") {
        @Override
        public SignRuleItemDto getRuleItemSub(int continueDay, int dayFeature, SignRuleConfigDto signRuleConfig) {
            continueDay += dayFeature;
            Calendar signCal = Calendar.getInstance();
            signCal.set(Calendar.DAY_OF_MONTH, signCal.get(Calendar.DAY_OF_MONTH) + (dayFeature - 1));

            int daysInMonth = signCal.getActualMaximum(Calendar.DAY_OF_MONTH);
            int daysOfMonth = signCal.get(Calendar.DAY_OF_MONTH);

            if (daysOfMonth == daysInMonth && continueDay >= daysInMonth &&signRuleConfig.getItems().size() == 2) {//签到月的最后一天
                    return signRuleConfig.getItems().get(1);
            }
            if (continueDay > daysOfMonth) {
                continueDay = daysOfMonth;
            }
            SignRuleItemDto item = signRuleConfig.getItems().get(0);
            SignRuleItemDto itemResult = BeanUtils.copy(item, SignRuleItemDto.class);
            itemResult.setCredits(itemResult.getCredits() + (continueDay - 1) * signRuleConfig.getIncr());
            return itemResult;
        }
    };

    private static Map<Integer, SignTypeEnum> typeEnumMap = new HashMap<>();

    static {
        for (SignTypeEnum typeEnum : SignTypeEnum.values()) {
            typeEnumMap.put(typeEnum.getCode(), typeEnum);
        }
    }

    /**
     * 通过code获取签到类型
     *
     * @param code
     * @return
     */
    public static SignTypeEnum getByCode(int code) {
        SignTypeEnum type = typeEnumMap.get(code);
        if (type == null) {
            throw new ActivityCenterRuntimeException("不支持的签到类型");
        }
        return type;
    }

    private Integer code;
    private String desc;

    SignTypeEnum(Integer code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    /**
     * @param continueDay    截止到昨天的连续签到天数（包含昨天）
     * @param dayFeature     继续签到的天数
     * @param signRuleConfig
     * @return
     */
    protected abstract SignRuleItemDto getRuleItemSub(int continueDay, int dayFeature, SignRuleConfigDto signRuleConfig);

    /**
     * 根据签到规则计算连续签到第continueDay天所适用的签到奖励条目
     *
     * @param continueDay    截止到昨天的连续签到天数（包含昨天）
     * @param signDate       目标签到日期
     * @param signRuleConfig 签到规则
     * @return
     */
    public SignRuleItemDto getRuleItem(Integer continueDay, Date signDate, SignRuleConfigDto signRuleConfig) {
        int dayFeature;
        if (signDate == null || (dayFeature = DateUtils.daysBetween(new Date(), signDate)) < 0) {
            throw new ActivityCenterException("目标签到日期非法");
        }
        return this.getRuleItemSub(continueDay, dayFeature + 1, signRuleConfig);
    }

    public Integer getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }
}
