/**
 * Project Name:activity-center-biz File Name:GameCenterResourceServiceImpl.java Package
 * Name:cn.com.duiba.activity.center.biz.service.gamecenter.impl Date:2016年10月20日下午3:33:31 Copyright (c) 2016,
 * duiba.com.cn All Rights Reserved.
 */

package cn.com.duiba.activity.center.biz.service.gamecenter.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Resource;

import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import cn.com.duiba.activity.center.api.dto.gamecenter.DefinedActivityTypesDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.GameCenterStatisticDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.LittleGameResourceDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.LittleGameResourceListDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.RelatedActivityDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.RelatedActivityListDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.ResourceLocationNamesDto;
import cn.com.duiba.activity.center.api.dto.gamecenter.ShieldItemDto;
import cn.com.duiba.activity.center.biz.dao.gamecenter.GameCenterResourceDao;
import cn.com.duiba.activity.center.biz.dao.gamecenter.GameCenterResourceLocationDao;
import cn.com.duiba.activity.center.biz.entity.gamecenter.GameCenterResourceEntity;
import cn.com.duiba.activity.center.biz.entity.gamecenter.GameCenterResourceLocationEntity;
import cn.com.duiba.activity.center.biz.entity.littlegame.LittleGameEntity;
import cn.com.duiba.activity.center.biz.service.gamecenter.ActivityShieldService;
import cn.com.duiba.activity.center.biz.service.gamecenter.GameCenterException;
import cn.com.duiba.activity.center.biz.service.gamecenter.GameCenterResourceService;
import cn.com.duiba.activity.center.biz.service.gamecenter.GeneralActivityItem;
import cn.com.duiba.activity.center.biz.service.gamecenter.MultiActivityQueryingService;
import cn.com.duiba.activity.center.biz.service.littlegame.LittleGameService;
import cn.com.duiba.wolf.utils.BeanUtils;

/**
 * ClassName:GameCenterResourceServiceImpl <br/>
 * Function: 游戏中心Service层实现. <br/>
 * Reason: 游戏中心Service层实现. <br/>
 * Date: 2016年10月20日 下午3:33:31 <br/>
 *
 * @author Zhuzhiyong
 * @version
 * @see
 * @since JDK 1.6
 */
@Service
public class GameCenterResourceServiceImpl implements GameCenterResourceService {

    private static final Logger           log              = LoggerFactory.getLogger(GameCenterResourceServiceImpl.class);

    @Resource
    private GameCenterResourceDao         gameCenterResourceDao;
    @Resource
    private GameCenterResourceLocationDao gameCenterResourceLocationDao;
    @Resource
    private MultiActivityQueryingService  multiActivityQueryingService;
    @Resource
    private ActivityShieldService         gameCenterShieldService;
    @Resource
    private LittleGameService             littleGameService;

    /** 资源正常状态. */
    private static final byte             RESOURCE_NORMAL  = (byte) 1;

    /** 资源删除状态. */
    private static final byte             RESOURCE_DELETED = (byte) 2;

    /**
     * Find statistic.
     *
     * @return the game center statistic dto
     * @throws GameCenterException the game center exception
     */
    @Override
    public GameCenterStatisticDto findStatistic() throws GameCenterException {

        GameCenterStatisticDto littleGameResourceDtoList = new GameCenterStatisticDto();
        Integer littleGameCount = littleGameService.selectCountByStatus(1); // 开启且可见的游戏
        littleGameResourceDtoList.setLittleGameCount(littleGameCount);

        Long bannerLocId = findLocationIdByName(ResourceLocationNamesDto.BANNER);

        List<GameCenterResourceEntity> bannerResourceEntities = gameCenterResourceDao.findResourceByLocationId(bannerLocId);
        // Banner：游戏
        littleGameResourceDtoList.setBannerLittleGameCount(countLittleGame(bannerResourceEntities));

        // TODO Banner：活动

        // 今日推荐
        Long todayRecommendLocId = findLocationIdByName(ResourceLocationNamesDto.TODAY_RECOMMEND);
        List<GameCenterResourceEntity> todayRecommendResEntities = gameCenterResourceDao.findResourceByLocationId(todayRecommendLocId);
        littleGameResourceDtoList.setTodayRecommendCount(countLittleGame(todayRecommendResEntities));

        // 大家都在玩
        Long everyonePalyLocId = findLocationIdByName(ResourceLocationNamesDto.EVERYONE_PLAY);
        List<GameCenterResourceEntity> everyonePlayResEntities = gameCenterResourceDao.findResourceByLocationId(everyonePalyLocId);

        littleGameResourceDtoList.setEveryonePlayCount(countLittleGame(everyonePlayResEntities));
        return littleGameResourceDtoList;
    }

    private int countLittleGame(List<GameCenterResourceEntity> gameCenterResourceEntities) {
        if (CollectionUtils.isEmpty(gameCenterResourceEntities)) {
            return 0;
        }
        int count = 0;
        for (GameCenterResourceEntity entity : gameCenterResourceEntities) {
            // 只统计小游戏数量
            if (entity.getActivityType() != DefinedActivityTypesDto.LITTLE_GAME) {
                continue;
            }
            LittleGameEntity littleGameEntity = littleGameService.selectById(entity.getActivityId());
            if (!Utils.filterLittleGameByStatus(littleGameEntity, entity.getActivityId())) {
                count++;
            }
        }
        return count;
    }

    /**
     * Find little game resource.
     *
     * @param offset the offset
     * @param max the max
     * @return the list
     */
    @Override
    public LittleGameResourceListDto findLittleGameResource(Integer offset, Integer max) {
        LittleGameResourceListDto littleGameResourceListDto = new LittleGameResourceListDto();
        littleGameResourceListDto.setCount(littleGameService.selectCountByStatus(1)); // 开启且可见的游戏

        List<LittleGameEntity> entities = littleGameService.selectListByStatus(1, offset, max);
        if (CollectionUtils.isEmpty(entities)) {
            return littleGameResourceListDto;
        }
        List<LittleGameResourceDto> littleGameResourceDtoList = new ArrayList<LittleGameResourceDto>();
        LittleGameResourceDto temp;
        for (LittleGameEntity entity : entities) {
            temp = new LittleGameResourceDto();
            temp.setActivityId(entity.getId());
            temp.setIconImg(entity.getLittleGameIconImg());
            temp.setGmtCreate(entity.getGmtCreate());
            // 先填充时间，如果有关联在游戏中心手机展示，则以最后更新时间为准
            temp.setGmtModified(entity.getGmtModified());
            setLocationsAndUpdateTime(temp);
            littleGameResourceDtoList.add(temp);
        }
        littleGameResourceListDto.setLittleGameResourceDtoList(littleGameResourceDtoList);
        return littleGameResourceListDto;
    }

    /**
     * Update resource location.
     *
     * @param activityId the activity id
     * @param activityType the activity type
     * @param addLocationNames the add location names
     * @param delLocationNames the del location names
     * @return the list
     * @throws GameCenterException the game center exception
     */
    @Override
    public Boolean updateResourceLocation(Long activityId, Integer activityType, List<String> addLocationNames,
                                          List<String> delLocationNames) throws GameCenterException {
        // 既没有新增又没有删除
        if (CollectionUtils.isEmpty(addLocationNames) && CollectionUtils.isEmpty(delLocationNames)) {
            return true;
        }
        boolean result = true;
        if (!CollectionUtils.isEmpty(addLocationNames)) {
            for (String name : addLocationNames) {
                result = result && createGameCenterResource(activityType, activityId, name);
            }
        }

        if (!CollectionUtils.isEmpty(delLocationNames)) {
            for (String name : delLocationNames) {
                result = result && deleteGameCenterResource(activityType, activityId, name);
            }
        }
        return result;
    }

    private boolean createGameCenterResource(Integer activityType, Long activityId,
                                             String name) throws GameCenterException {
        Long locId = findLocationIdByName(name);

        // 插入前先看记录是否存在，存在更新状态
        GameCenterResourceEntity daoEntity = gameCenterResourceDao.findResource(activityType, activityId, locId);
        if (daoEntity != null) {
            return gameCenterResourceDao.updateStatus(daoEntity.getId(), RESOURCE_NORMAL) == 1;
        }

        GameCenterResourceEntity entity = new GameCenterResourceEntity();

        entity.setActivityId(activityId);
        entity.setActivityType(activityType);
        entity.setResourceLocationId(locId);
        entity.setStatus(RESOURCE_NORMAL);
        // 默认优先级0
        entity.setPayload(0);

        Date now = new Date();
        entity.setGmtCreate(now);
        entity.setGmtModified(now);
        return gameCenterResourceDao.insert(entity) == 1;
    }

    private boolean deleteGameCenterResource(Integer activityType, Long activityId,
                                             String name) throws GameCenterException {
        Long locId = findLocationIdByName(name);
        // 插入前先看记录是否存在，存在更新状态
        GameCenterResourceEntity daoEntity = gameCenterResourceDao.findResource(activityType, activityId, locId);
        if (daoEntity == null) {
            // 不存在也认为是成功的
            return true;
        }
        return gameCenterResourceDao.updateStatus(daoEntity.getId(), RESOURCE_DELETED) == 1;
    }

    /**
     * Find related activity.
     *
     * @param offset the offset
     * @param max the max
     * @return the list
     * @throws GameCenterException
     */
    @Override
    public RelatedActivityListDto findRelatedActivity(Integer offset, Integer max) throws GameCenterException {

        RelatedActivityListDto relatedActivityListDto = new RelatedActivityListDto();
        relatedActivityListDto.setCount(gameCenterResourceDao.countResourceExcludeActivityType(DefinedActivityTypesDto.LITTLE_GAME));

        // 查询除小游戏以外的活动
        List<GameCenterResourceEntity> entities = gameCenterResourceDao.findResourceExcludeActivityType(DefinedActivityTypesDto.LITTLE_GAME,
                                                                                                        offset, max);
        if (CollectionUtils.isEmpty(entities)) {
            return relatedActivityListDto;
        }
        List<RelatedActivityDto> relatedActivityDtoList = new ArrayList<RelatedActivityDto>();
        GeneralActivityItem item;
        for (GameCenterResourceEntity entity : entities) {
            item = multiActivityQueryingService.findActivityByTypeAndId(entity.getActivityType(),
                                                                        entity.getActivityId());
            relatedActivityDtoList.add(BeanUtils.copy(item, RelatedActivityDto.class));
        }
        relatedActivityListDto.setRelatedActivityDtoList(relatedActivityDtoList);
        return relatedActivityListDto;
    }

    /**
     * Adds the related activity.
     *
     * @param activityId the activity id
     * @param activityType the activity type
     * @return the boolean
     * @throws GameCenterException
     */
    @Override
    public Boolean addRelatedActivity(Long activityId, Integer activityType) throws GameCenterException {

        // TODO 检查活动存在与否？状态？

        List<GameCenterResourceEntity> entities = gameCenterResourceDao.findResourceByActivityTypeAndIdForEdit(activityType,
                                                                                                               activityId);
        GameCenterResourceEntity entity;
        Date now = new Date();
        if (CollectionUtils.isEmpty(entities)) {
            entity = new GameCenterResourceEntity();
            entity.setActivityId(activityId);
            entity.setActivityType(activityType);
            entity.setPayload(0);
            entity.setResourceLocationId(findLocationIdByName(ResourceLocationNamesDto.BANNER));
            entity.setStatus(RESOURCE_NORMAL);
            entity.setGmtCreate(now);
            entity.setGmtModified(now);
            return gameCenterResourceDao.insert(entity) == 1;
        }

        // 当前活动只可能在banner区域，list最多只有一个元素
        entity = entities.get(0);
        // 已经存在
        if (entity.getStatus().byteValue() == RESOURCE_NORMAL) {
            log.error(">>addRelatedActivity: resource does exist, nothing need to do.");
            return false;
        }
        return gameCenterResourceDao.updateStatus(entity.getId(), RESOURCE_NORMAL) == 1;
    }

    /**
     * Delete related activity.
     *
     * @param activityId the activity id
     * @param activityType the activity type
     * @return the boolean
     */
    @Override
    public Boolean deleteRelatedActivity(Long activityId, Integer activityType) {
        List<GameCenterResourceEntity> entities = gameCenterResourceDao.findResourceByActivityTypeAndIdForEdit(activityType,
                                                                                                               activityId);
        if (CollectionUtils.isEmpty(entities)) {
            log.error(">>deleteRelatedActivity: resource does exist, nothing need to do.");
            return false;
        }
        // 当前活动只可能在banner区域，list最多只有一个元素
        GameCenterResourceEntity entity = entities.get(0);
        if (entity.getStatus().byteValue() == RESOURCE_DELETED) {
            log.error(">>deleteRelatedActivity: resource has deleted, nothing need to do.");
            return false;
        }
        return gameCenterResourceDao.updateStatus(entity.getId(), RESOURCE_DELETED) == 1;
    }

    /**
     * 开发者端查询小游戏列表.
     *
     * @param appId the app id
     * @param offset the offset
     * @param max the max
     * @return the list
     */
    @Override
    public LittleGameResourceListDto findAvailableLittleGameByAppId(Long appId, Integer offset, Integer max) {
        LittleGameResourceListDto littleGameResourceListDto = new LittleGameResourceListDto();
        List<GameCenterResourceEntity> entities = gameCenterResourceDao.findResourceByActivityType(DefinedActivityTypesDto.LITTLE_GAME);
        if (CollectionUtils.isEmpty(entities)) {
            return littleGameResourceListDto;
        }

        List<LittleGameResourceDto> littleGameResourceDtoList = new ArrayList<LittleGameResourceDto>();
        LittleGameResourceDto temp;
        for (GameCenterResourceEntity entity : entities) {
            // 已经添加
            temp = containsLittleGame(littleGameResourceDtoList, entity.getActivityId());
            if (temp != null) {
                // 最后更新时间:对于关联在多个区域的情况，取最后更新时间
                temp.setGmtModified(Utils.latter(temp.getGmtModified(), entity.getGmtModified()));
                continue;
            }

            LittleGameEntity littleGameEntity = littleGameService.selectById(entity.getActivityId());
            if (Utils.filterLittleGameByStatus(littleGameEntity, entity.getActivityId())) {
                continue;
            }

            // 剔除开发者屏蔽小游戏
            if (gameCenterShieldService.findShield(appId, DefinedActivityTypesDto.LITTLE_GAME,
                                                   entity.getActivityId()) != null) {
                log.debug(">>findAvailableLittleGameByAppId: littleGame was filtered by shield. appid is {}, littleGameId is {}",
                          appId, entity.getActivityId());
                continue;
            }

            temp = new LittleGameResourceDto();
            temp.setActivityId(entity.getActivityId());
            temp.setGmtCreate(entity.getGmtCreate());
            temp.setGmtModified(entity.getGmtModified());
            temp.setTitle(littleGameEntity.getLittleGameTitle());
            temp.setIconImg(littleGameEntity.getLittleGameIconImg());
            littleGameResourceDtoList.add(temp);
        }

        if (littleGameResourceDtoList.isEmpty()) {
            return littleGameResourceListDto;
        }

        // 按修改时间倒序
        Collections.sort(littleGameResourceDtoList, new Comparator<LittleGameResourceDto>() {

            @Override
            public int compare(LittleGameResourceDto o1, LittleGameResourceDto o2) {
                return o1.getGmtModified().compareTo(o2.getGmtModified());
            }
        });
        Collections.reverse(littleGameResourceDtoList);

        littleGameResourceListDto.setCount(littleGameResourceDtoList.size());
        littleGameResourceListDto.setLittleGameResourceDtoList(Utils.subList(littleGameResourceDtoList, offset, max));
        return littleGameResourceListDto;
    }

    /**
     * Find hidden little game by app id.
     *
     * @param appId the app id
     * @param offset the offset
     * @param max the max
     * @return the list
     */
    @Override
    public LittleGameResourceListDto findHiddenLittleGameByAppId(Long appId, Integer offset, Integer max) {

        LittleGameResourceListDto littleGameResourceListDto = new LittleGameResourceListDto();
        Map<String, Object> paramMap = new HashMap<String, Object>();
        paramMap.put("appId", appId);
        paramMap.put("activityType", DefinedActivityTypesDto.LITTLE_GAME);

        List<ShieldItemDto> shields = gameCenterShieldService.findShieldList(paramMap);
        if (CollectionUtils.isEmpty(shields)) {
            return littleGameResourceListDto;
        }
        List<LittleGameResourceDto> littleGameResourceDtoList = new ArrayList<LittleGameResourceDto>();
        LittleGameResourceDto temp;
        for (ShieldItemDto item : shields) {
            LittleGameEntity littleGameEntity = littleGameService.selectById(item.getActivityId());
            if (Utils.filterLittleGameByStatus(littleGameEntity, item.getActivityId())) {
                continue;
            }

            temp = new LittleGameResourceDto();
            temp.setShieldId(item.getId());
            temp.setActivityType(DefinedActivityTypesDto.LITTLE_GAME);
            temp.setActivityId(item.getActivityId());
            temp.setGmtCreate(item.getGmtCreate());
            temp.setGmtModified(item.getGmtModified());
            temp.setTitle(littleGameEntity.getLittleGameTitle());
            temp.setIconImg(littleGameEntity.getLittleGameIconImg());
            littleGameResourceDtoList.add(temp);
        }

        if (littleGameResourceDtoList.isEmpty()) {
            return littleGameResourceListDto;
        }

        littleGameResourceListDto.setCount(littleGameResourceDtoList.size());
        littleGameResourceListDto.setLittleGameResourceDtoList(Utils.subList(littleGameResourceDtoList, offset, max));
        return littleGameResourceListDto;
    }

    /**
     * Find location id by name.
     *
     * @param name the name
     * @return the long
     * @throws GameCenterException the game center exception
     */
    private Long findLocationIdByName(String name) throws GameCenterException {
        GameCenterResourceLocationEntity entity = gameCenterResourceLocationDao.findResourceLocationByName(name);
        if (entity == null) {
            throw new GameCenterException("No this location found: " + name);
        }
        return entity.getId();
    }

    private void setLocationsAndUpdateTime(LittleGameResourceDto littleGameResourceDto) {
        List<GameCenterResourceEntity> entities = gameCenterResourceDao.findResourceByActivityTypeAndIdForEdit(littleGameResourceDto.getActivityType(),
                                                                                                               littleGameResourceDto.getActivityId());
        if (CollectionUtils.isEmpty(entities)) {
            return;
        }

        List<String> locations = new ArrayList<String>();
        List<Date> gmtModifiedList = new ArrayList<Date>();
        GameCenterResourceLocationEntity gameCenterResourceLocationEntity;
        for (GameCenterResourceEntity entity : entities) {
            gameCenterResourceLocationEntity = gameCenterResourceLocationDao.find(entity.getResourceLocationId());
            if (gameCenterResourceLocationEntity == null) {
                log.error(">>findLocationsByActivityTypeAndId: no resource location found: "
                          + entity.getResourceLocationId());
                continue;
            }
            gmtModifiedList.add(entity.getGmtModified());
            locations.add(gameCenterResourceLocationEntity.getTitle());
        }

        littleGameResourceDto.setLocations(locations);

        // 资源表中最后更新的资源位设置为最后更新时间
        if (!gmtModifiedList.isEmpty()) {
            Collections.sort(gmtModifiedList);
            Collections.reverse(gmtModifiedList);
            littleGameResourceDto.setGmtModified(gmtModifiedList.get(0));
        }
    }

    private LittleGameResourceDto containsLittleGame(List<LittleGameResourceDto> list, Long activityId) {
        if (CollectionUtils.isEmpty(list)) {
            return null;
        }
        for (LittleGameResourceDto item : list) {
            if (Objects.equals(item.getActivityId(), activityId)) {
                return item;
            }
        }
        return null;
    }
}
