package cn.com.duiba.activity.center.biz.service.rob.impl;

import cn.com.duiba.activity.center.api.dto.rob.TodayRobConfigDto;
import cn.com.duiba.activity.center.api.dto.rob.category.RobActivityDto;
import cn.com.duiba.activity.center.api.dto.rob.category.RobCategoryDetailDto;
import cn.com.duiba.activity.center.biz.dao.DsConstants;
import cn.com.duiba.activity.center.biz.dao.rob.TodayRobConfigDao;
import cn.com.duiba.activity.center.biz.dao.rob_category.RobBarCategoryRelationDao;
import cn.com.duiba.activity.center.biz.dao.rob_category.RobCategoryActivityRelationDao;
import cn.com.duiba.activity.center.biz.dao.rob_category.RobCategoryBarAppRelationDao;
import cn.com.duiba.activity.center.biz.dao.rob_category.RobCategoryDao;
import cn.com.duiba.activity.center.biz.entity.robcategory.RobCategoryBarAppRelationEntity;
import cn.com.duiba.activity.center.biz.entity.robcategory.RobCategoryEntity;
import cn.com.duiba.activity.center.biz.service.rob.RobCategoryService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;

/**
 * 描述:今日必抢分类相关逻辑操作实现类
 *
 * @author :sunchangji
 * @create :2016-10-17 17:40
 */
@Service
public class RobCategoryServiceImpl implements RobCategoryService {
    private static final Logger log = LoggerFactory.getLogger(RobCategoryBarServiceImpl.class);

    private static final Long DEFAULT_BAR_ID = 1L;//共享默认分类栏id
    @Autowired
    private RobCategoryBarAppRelationDao robCategoryBarAppRelationDao;
    @Autowired
    private RobBarCategoryRelationDao robBarCategoryRelationDao;
    @Autowired
    private RobCategoryActivityRelationDao robCategoryActivityRelationDao;
    @Autowired
    private RobCategoryDao robCategoryDao;
    @Autowired
    private TodayRobConfigDao todayRobConfigDao;

    @Override
    public List<RobCategoryEntity> getAllCategory() {
        return robCategoryDao.getAllCategory();
    }

    public List<RobCategoryEntity> getByAppId(Long appId) {
        RobCategoryBarAppRelationEntity entity = robCategoryBarAppRelationDao.getByAppId(appId);
        Long barId = null;
        if (entity == null) {
            barId = DEFAULT_BAR_ID;
        } else {
            barId = entity.getBarId();
        }

        final List<Long> categoryIds = robBarCategoryRelationDao.getRobCategoryIdsByBarId(barId);
        if (CollectionUtils.isEmpty(categoryIds)) {
            log.error("分类栏没有配置今日必抢分类");
            return Collections.emptyList();
        }

        List<RobCategoryEntity> robCategoryEntityList = robCategoryDao.getRobCategoryByIds(categoryIds);
        if (CollectionUtils.isEmpty(categoryIds)) {
            log.error("今日必抢分类不存在");
            return Collections.emptyList();
        }
        //分类列表排序
        Collections.sort(robCategoryEntityList, new Comparator<RobCategoryEntity>() {
            @Override
            public int compare(RobCategoryEntity o1, RobCategoryEntity o2) {
                if (categoryIds.indexOf(o1.getId()) > categoryIds.indexOf(o2.getId())) {
                    return 1;
                } else if (categoryIds.indexOf(o1.getId()) < categoryIds.indexOf(o2.getId())) {
                    return -1;
                } else {
                    return 0;
                }
            }
        });
        return robCategoryEntityList;
    }

    @Override
    public List<TodayRobConfigDto> getByCategoryId(Long categoryId, int offset, int pageSize) {
        final List<Long> robActivityIds = robCategoryActivityRelationDao.getPageRobActivityIdsByCategoryId(categoryId, offset, pageSize);
        List<TodayRobConfigDto> todayRobConfigDtoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(robActivityIds))
            return todayRobConfigDtoList;
        todayRobConfigDtoList = todayRobConfigDao.findTodayRobConfigForAppByIds(robActivityIds);
        if (CollectionUtils.isEmpty(todayRobConfigDtoList))
            return todayRobConfigDtoList;
        //结果排序
        todayRobConfigDtoListSort(robActivityIds, todayRobConfigDtoList);
        return todayRobConfigDtoList;
    }

    /**
     * 对今日必抢活动按in中集合的顺序排序
     *
     * @param robActivityIds        in中的集合
     * @param todayRobConfigDtoList 要排序的活动集合
     */
    private void todayRobConfigDtoListSort(final List<Long> robActivityIds, List<TodayRobConfigDto> todayRobConfigDtoList) {
        //结果排序
        Collections.sort(todayRobConfigDtoList, new Comparator<TodayRobConfigDto>() {
            @Override
            public int compare(TodayRobConfigDto o1, TodayRobConfigDto o2) {
                if (robActivityIds.indexOf(o1.getId()) > robActivityIds.indexOf(o2.getId())) {
                    return 1;
                } else if (robActivityIds.indexOf(o1.getId()) < robActivityIds.indexOf(o2.getId())) {
                    return -1;
                } else {
                    return 0;
                }
            }
        });

    }

    @Override
    public List<RobActivityDto> getForManageByCategoryId(Long categoryId) {
        List<Long> activityIds = robCategoryActivityRelationDao.getAllRobActivityIdsByCategoryId(categoryId);
        List<RobActivityDto> robActivityDtoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(activityIds)) {
            return robActivityDtoList;
        }
        int totalCount = activityIds.size();
        //如果分类下的活动大于50个，则分批查找，避免in的过长导致查询性能问题
        if (totalCount > 50) {
            int pageSize = 50;
            for (int i = 0; i < totalCount; i += pageSize) {
                List<Long> newlist = null;
                if ((i + pageSize) > totalCount) {
                    newlist = activityIds.subList(i, totalCount);
                } else {
                    newlist = activityIds.subList(i, i + pageSize);
                }
                List<RobActivityDto> robActivityDtos = todayRobConfigDao.findTodayRobConfigForManagerByIds(newlist);
                if (robActivityDtos != null)
                    robActivityDtoList.addAll(robActivityDtos);
            }
        }else{
           robActivityDtoList = todayRobConfigDao.findTodayRobConfigForManagerByIds(activityIds);
        }
        if (robActivityDtoList.isEmpty()) {
            return robActivityDtoList;
        }
        //结果排序
        robActivityDtoListSort(activityIds, robActivityDtoList);

        return robActivityDtoList;
    }

    /**
     * 对RobActivityDtoList集合按activityIds顺序排序
     *
     * @param activityIds
     * @param robActivityDtoList
     */
    private void robActivityDtoListSort(final List<Long> activityIds, List<RobActivityDto> robActivityDtoList) {
        Collections.sort(robActivityDtoList, new Comparator<RobActivityDto>() {
            @Override
            public int compare(RobActivityDto o1, RobActivityDto o2) {
                if (activityIds.indexOf(o1.getId()) > activityIds.indexOf(o2.getId())) {
                    return 1;
                } else if (activityIds.indexOf(o1.getId()) < activityIds.indexOf(o2.getId())) {
                    return -1;
                } else {
                    return 0;
                }
            }
        });
    }

    @Override
    public List<RobCategoryEntity> getCategoryPage(String name, String title, Integer offset, Integer pageSize) {
        if (offset == null || offset < 0)
            offset = 0;
        if (pageSize == null || pageSize < 1)
            pageSize = 20;

        return robCategoryDao.getPageRobCategory(name, title, offset, pageSize);
    }

    @Override
    public Integer countByNameAndTitle(String name, String title) {
        return robCategoryDao.countByNameAndTitle(name, title);
    }

    @Override
    @Transactional(value = DsConstants.DATABASE_CREDITS_ACTIVITY)
    public Boolean deleteById(Long id) {
        if (id.equals(1L)) {
            log.error("默认[全部]分类禁止删除");
            return false;
        }
        robBarCategoryRelationDao.deleteByCategoryId(id);
        robCategoryActivityRelationDao.deleteByCategoryId(id);
        robCategoryDao.deleteById(id);
        return true;
    }

    @Override
    @Transactional(value = DsConstants.DATABASE_CREDITS_ACTIVITY)
    public Boolean insertCategory(RobCategoryDetailDto robCategoryDetailDto) {
        RobCategoryEntity robCategoryEntity = new RobCategoryEntity(robCategoryDetailDto.getName(), robCategoryDetailDto.getTitle());
        Long categoryId = robCategoryDao.insertRobCategory(robCategoryEntity);
        if (categoryId == null || categoryId < 1)
            return false;
        Map<Long, Integer> activitIdAndPayloadMap = robCategoryDetailDto.getActivitIdAndPayloadMap();
        if (!CollectionUtils.isEmpty(activitIdAndPayloadMap)) {
            for (Map.Entry<Long, Integer> entry : activitIdAndPayloadMap.entrySet()) {
                robCategoryActivityRelationDao.insertRobCategoryActivityRelation(categoryId, entry.getKey(), entry.getValue());
            }
        }

        return true;
    }

    @Override
    @Transactional(value = DsConstants.DATABASE_CREDITS_ACTIVITY)
    public Boolean batchDeleteCategoryActivityRelation(Long categoryId, List<Long> activityIds) {
        for (Long activityId : activityIds) {
            robCategoryActivityRelationDao.deleteByActivityIdAndCategoryId(activityId, categoryId);
        }
        return true;
    }

    @Override
    @Transactional(value = DsConstants.DATABASE_CREDITS_ACTIVITY)
    public Boolean batchUpdateRobCategoryActivityRelationPayload(RobCategoryDetailDto robCategoryDetailDto) {
        Long categoryId = robCategoryDetailDto.getCategoryId();
        String name = robCategoryDetailDto.getName();
        String title = robCategoryDetailDto.getTitle();
        robCategoryDao.updateRobCategory(categoryId, name, title);
        Map<Long, Integer> activitIdAndPayloadMap = robCategoryDetailDto.getActivitIdAndPayloadMap();
        if(activitIdAndPayloadMap!=null){
            for (Map.Entry<Long, Integer> entry : activitIdAndPayloadMap.entrySet()) {
                robCategoryActivityRelationDao.updateRobCategoryActivityRelation(categoryId, entry.getKey(), entry.getValue());
            }
        }
        return true;
    }

    @Override
    public List<RobCategoryEntity> getRobCategorysByIds(List<Long> categoryIdList) {
        if(CollectionUtils.isEmpty(categoryIdList))
            return Collections.emptyList();
        return robCategoryDao.getRobCategorysByIds(categoryIdList);
    }

    @Override
    public RobCategoryEntity getCategoryById(Long id) {
        return robCategoryDao.getCategoryById(id);
    }

}
