package cn.com.duiba.anticheat.center.biz.remoteservice.risk.impl;

import cn.com.duiba.anticheat.center.api.dto.RiskRuleEngineResultDto;
import cn.com.duiba.anticheat.center.api.enums.RiskDecisionEnum;
import cn.com.duiba.anticheat.center.api.param.RiskProjectxParam;
import cn.com.duiba.anticheat.center.api.remoteservice.risk.RemoteRiskProjectXService;
import cn.com.duiba.anticheat.center.biz.config.RiskActivityJoinConfig;
import cn.com.duiba.anticheat.center.biz.constant.RedisKeyFactory;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.wolf.cache.AdvancedCacheClient;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RestController;

import javax.annotation.Resource;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author JunAngLiu
 * @Title: RemoteRiskProjectXServiceImpl
 * @Description:
 * @date 2019/9/2513:58
 */
@RestController
public class RemoteRiskProjectXServiceImpl implements RemoteRiskProjectXService {

    private static final Logger LOGGER = LoggerFactory.getLogger(RemoteRiskProjectXServiceImpl.class);

    //同一用户ID对应的IP数量上限
    private static final int IP_LIMIT = 6;
    //同一用户ID对应的UA数量上限
    private static final int UA_LIMIT = 5;
    //同一IP对应的用户id数量上限
    private static final int CID_LIMIT = 60;

    //星速台活动风控开启状态
    private static final int OFF_SWITCH = 0;
    //ua 风控规则中 关键字
    private static final String UA_WINDOWS = "windows";


    @Resource(name = "redisTemplate")
    private AdvancedCacheClient advancedCacheClient;
    @Autowired
    private RiskActivityJoinConfig riskActivityJoinConfig;



    @Override
    public void log(RiskProjectxParam param) throws BizException {
        if(param == null || param.getConsumerId() == null || param.getProjectId() == null ||param.getIp() == null){
            throw new BizException("参数异常");
        }
        if (LOGGER.isInfoEnabled()) {
            LOGGER.info("星速台活动风控日志,参数={}", JSONObject.toJSONString(param));
        }

        try{
            //记录同一用户ID对应的IP数量
            this.redisLog(this.getIpKey(param.getProjectId(),param.getConsumerId()),param.getIp(),IP_LIMIT);
            //记录同一用户ID对应的UA数量上限
            if(StringUtils.isNotBlank(param.getUa())){
                String converUa = DigestUtils.md5Hex(param.getUa().toLowerCase());
                this.redisLog(this.getUAKey(param.getProjectId(),param.getConsumerId()),converUa,UA_LIMIT);
            }
            //记录同一IP对应的用户id数量上限
            this.redisLog(this.getCidKey(param.getProjectId(),param.getIp()),String.valueOf(param.getConsumerId()),CID_LIMIT);
        } catch (Exception e){
            LOGGER.warn("星速台活动风控日志记录异常,param = {}",JSONObject.toJSONString(param));
        }

    }



    /**
     * 将当天 用户数据记录到 redis
     * @param key
     * @param value
     * @param limit
     */
    private void redisLog(String key,String value,int limit){
        Object obj = advancedCacheClient.get(key);
        //获取当天剩余时间
        LocalDateTime midnight = LocalDateTime.now().plusDays(1).withHour(0).withMinute(0).withSecond(0).withNano(0);
        long seconds = ChronoUnit.SECONDS.between(LocalDateTime.now(), midnight);

        if(Objects.isNull(obj)){
            advancedCacheClient.set(key,Lists.newArrayList(value),Long.valueOf(seconds).intValue(), TimeUnit.SECONDS);
        }else {

            List<String> values = this.convertObjToList(obj);
            //如果数量大于限制，就不记录了
            if(!values.contains(value) && values.size() < limit + 1 ){
                values.add(value);
                advancedCacheClient.set(key,values,Long.valueOf(seconds).intValue(), TimeUnit.SECONDS);
                return;
            }
        }
    }










    @Override
    public RiskRuleEngineResultDto execute(RiskProjectxParam param) throws BizException {

        if(param == null || param.getConsumerId() == null || param.getProjectId() == null ||param.getIp() == null){
            throw new BizException("参数异常");
        }
        RiskRuleEngineResultDto dto = new RiskRuleEngineResultDto();
        dto.setDecision(RiskDecisionEnum.ACCEPT);
        try{
            //1 开关
            if(riskActivityJoinConfig.getProjectxOpenSwitch() != null && OFF_SWITCH == riskActivityJoinConfig.getProjectxOpenSwitch()){
                return dto;
            }
            //2 先验证windows
            if(StringUtils.isNotBlank(param.getUa()) && param.getUa().toLowerCase().contains(UA_WINDOWS)){
                dto.setDecision(RiskDecisionEnum.REJECT);
                return dto;
            }

            //3 记录
            this.log(param);

            // 4 验证数量
            //对比同一用户ID对应的IP数量上限
            if(beyondLimit(this.getIpKey(param.getProjectId(),param.getConsumerId()),IP_LIMIT)){
                dto.setDecision(RiskDecisionEnum.REJECT);
                LOGGER.info("同一用户ID对应的IP数量上限拦截");
                return dto;
            }
            //对比同一用户ID对应的UA数量上限
            if(beyondLimit(this.getUAKey(param.getProjectId(),param.getConsumerId()),UA_LIMIT)){
                dto.setDecision(RiskDecisionEnum.REJECT);
                LOGGER.info("同一用户ID对应的UA数量上限拦截");
                return dto;
            }
            //对比同一IP对应的用户id数量上限
            if(beyondLimit(this.getCidKey(param.getProjectId(),param.getIp()),CID_LIMIT)){
                dto.setDecision(RiskDecisionEnum.REJECT);
                LOGGER.info("同一IP对应的用户id数量上限拦截");
                return dto;
            }

            return dto;
        }catch (BizException be){
            throw be;
        } catch (Exception e){
            LOGGER.warn("星速台活动风控 执行异常,param = {}",JSONObject.toJSONString(param));
            return dto;
        }

    }




    /**
     * 判断当前 是否超出限制
     * @param key
     * @param limit
     * @return
     */
    private boolean beyondLimit(String key,int limit){
        Object obj = advancedCacheClient.get(key);
        if(obj == null){
            return false;
        }
        List<String> values = this.convertObjToList(obj);
        if(values.size() > limit){
            return true;
        }
        return false;
    }




    //转换obj 为 list
    private List<String> convertObjToList(Object obj){
        if(obj instanceof ArrayList<?>) {
            List<String> values = Lists.newArrayList();
            for (Object o : (List<?>) obj) {
                values.add(String.class.cast(o));
            }
            return values;
        }
        return Collections.emptyList();
    }



    /**
     * 同一用户ID对应的IP key
     * @return
     */
    private String getIpKey(String projectId,Long consumerId){

        return RedisKeyFactory.K021.toString() + projectId + "_" + consumerId;
    }

    /**
     * 同一用户ID对应的UA key
     * @return
     */
    private String getUAKey(String projectId,Long consumerId){
        return RedisKeyFactory.K022.toString() + projectId + "_" + consumerId;
    }

    /**
     * 同一IP对应的用户ID数量 key
     * @return
     */
    private String getCidKey(String projectId,String ip){
        return RedisKeyFactory.K023.toString() + projectId + "_" + ip;
    }
}
