package cn.com.duiba.anticheat.center.biz.strategy.activity.impl;

import cn.com.duiba.anticheat.center.api.domain.activity.LotteryConsumerParams;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryOrderParams;
import cn.com.duiba.anticheat.center.api.domain.activity.LotteryRequestParams;
import cn.com.duiba.anticheat.center.biz.dao.activity.AnticheatLotteryDebugLogDao;
import cn.com.duiba.anticheat.center.biz.entity.activity.AnticheatLotteryDebugLogEntity;
import cn.com.duiba.anticheat.center.biz.entity.activity.AnticheatLotteryStrategyAppConfigEntity;
import cn.com.duiba.anticheat.center.biz.entity.activity.AnticheatLotteryStrategyConfigEntity;
import cn.com.duiba.anticheat.center.biz.strategy.activity.AnticheatLotteryStrategy;
import com.google.common.base.Objects;
import com.google.common.collect.HashMultimap;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.Map;
import java.util.Set;

/**
 * 在抽奖队列中,出现2个抽奖行为使用一个 deap 对应到2个不同的账号,那么,从该设备下第二个账号的第一次抽奖行为开始,命中本策略
 * 命中前,一个账号无限次抽奖不在考虑范围内
 * 命中后,从该设备下第二个账号第一次抽奖行为开始,全部算命中
 * @author Yanf Guo
 */
@Component
public class AnticheatLotterySameDeapStrategy implements AnticheatLotteryStrategy {

	/**
	 * key: operatingActivityId - ip
	 * value: Set<LotteryRecord>
	 */
	private static final HashMultimap<String, LotteryRecord> sameDeapMap = HashMultimap.create();

	/**
	 * 一个设备超过1个用户
	 */
	private static final int CONSUMER_DEAP_LIMIT_COUNT = 1;

	/**
	 * map key 的最大个数
	 */
	private static final int MAP_SIZE = 10000;

	@Autowired
	private AnticheatLotteryStrategyConfigService anticheatLotteryStrategyConfigService;

	@Autowired
	private AnticheatLotteryDebugLogDao anticheatLotteryDebugLogDao;

	@Override
	public int getEffectMode() {
		return anticheatLotteryStrategyConfigService.getCacheConfig(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_DEAP).getEffectMode();
	}

	@Override
	public boolean isEnable() {
		return anticheatLotteryStrategyConfigService.getCacheConfig(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_DEAP).getEnable();
	}

	@Override
	public boolean isAppEnable(Long appId) {
		Map<Long, AnticheatLotteryStrategyAppConfigEntity> map = anticheatLotteryStrategyConfigService.getCacheAppConfig(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_DEAP);
		return map.containsKey(appId);
	}

	@Override
	public AnticheatLotteryStrategyResult checkLottery(LotteryConsumerParams consumer, LotteryOrderParams order, LotteryRequestParams request) {
		if (StringUtils.isBlank(request.getDeap())) {	// 如果设备号不存在,默认不做验证
			return new AnticheatLotteryStrategyResult(false);
		}
		LotteryRecord record = new LotteryRecord();
		record.setDeap(request.getDeap());
		record.setOperatingActivityId(order.getOperatingActivityId());
		record.setDuibaActivityId(order.getDuibaActivityId());
		record.setTime(new Date());
		record.setType(order.getType());
		record.setLotteryOrderId(order.getLotteryOrderId());
		record.setConsumerId(consumer.getConsumerId());
		record.setAppId(consumer.getAppId());

		String key = getKey(order.getOperatingActivityId(), request.getDeap());
		if (sameDeapMap.size() > MAP_SIZE) {
			sameDeapMap.clear();
		}
		sameDeapMap.put(key, record);

		Set<LotteryRecord> records = sameDeapMap.get(key);
		if (records == null) {  // 由于并发导致 put 后被另一个线程 clear,则默认不命中
			return new AnticheatLotteryStrategyResult(false);
		}

		if (records.size() > CONSUMER_DEAP_LIMIT_COUNT) {
			Long debugId = doMatchProcess(consumer, order, request, records.size());
			return new AnticheatLotteryStrategyResult(true, debugId);
		}
		return new AnticheatLotteryStrategyResult(false);
	}

	private String getKey(Long operatingActivityId, String deap) {
		return String.format("%s-%s", operatingActivityId, deap);
	}

	private class LotteryRecord {
		private LotteryOrderParams.LotteryOrderTypeEnum type;
		private Long operatingActivityId;
		private Long duibaActivityId;
		private Date time;    // 当前时间
		private String deap;
		private Long lotteryOrderId;
		private Long consumerId;
		private Long appId;

		@Override
		public int hashCode() {
			return Objects.hashCode(consumerId);
		}

		@Override
		public boolean equals(Object obj) {
			if (!(obj instanceof LotteryRecord)) {
				return false;
			}
			LotteryRecord o = (LotteryRecord) obj;
			return this.consumerId.equals(o.consumerId);
		}

		public LotteryOrderParams.LotteryOrderTypeEnum getType() {
			return type;
		}

		public void setType(LotteryOrderParams.LotteryOrderTypeEnum type) {
			this.type = type;
		}

		public Long getOperatingActivityId() {
			return operatingActivityId;
		}

		public void setOperatingActivityId(Long operatingActivityId) {
			this.operatingActivityId = operatingActivityId;
		}

		public Long getDuibaActivityId() {
			return duibaActivityId;
		}

		public void setDuibaActivityId(Long duibaActivityId) {
			this.duibaActivityId = duibaActivityId;
		}

		public Date getTime() {
			return time;
		}

		public void setTime(Date time) {
			this.time = time;
		}

		public String getDeap() {
			return deap;
		}

		public void setDeap(String deap) {
			this.deap = deap;
		}

		public Long getLotteryOrderId() {
			return lotteryOrderId;
		}

		public void setLotteryOrderId(Long lotteryOrderId) {
			this.lotteryOrderId = lotteryOrderId;
		}

		public Long getConsumerId() {
			return consumerId;
		}

		public void setConsumerId(Long consumerId) {
			this.consumerId = consumerId;
		}

		public Long getAppId() {
			return appId;
		}

		public void setAppId(Long appId) {
			this.appId = appId;
		}
	}

	private Long doMatchProcess(LotteryConsumerParams consumer, LotteryOrderParams order, LotteryRequestParams request, int count) {
		AnticheatLotteryDebugLogEntity debug = new AnticheatLotteryDebugLogEntity();
		debug.setStrategyType(AnticheatLotteryStrategyConfigEntity.TYPE_SAME_DEAP);
		debug.setConsumerId(consumer.getConsumerId());
		debug.setPartnerUserId(consumer.getPartnerUserId());
		debug.setIp(request.getIp());
		debug.setLotteryOrderId(order.getLotteryOrderId());
		debug.setRelationId(order.getDuibaActivityId());
		debug.setRelationType(order.getType().value());
		debug.setOperatingActivityId(order.getOperatingActivityId());
		debug.setAppId(order.getAppId());
		debug.setMessage(String.format("此设备 (%s) 连续使用第 %s 个账号进行 %s 活动抽奖(app活动id: %s, duiba活动id(或app-单品抽奖id): %s)", request.getDeap(), count, order.getType().desc(), order.getOperatingActivityId(), order.getDuibaActivityId()));
		anticheatLotteryDebugLogDao.insert(debug);

		return debug.getId();
	}

}
