package cn.com.duiba.bigdata.common.biz.utils;

import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.*;

@Slf4j
public class FutureTaskUtil {

    /**
     * 异步提交需要执行的任务
     *
     * @param executorService 线程池
     * @param callable        具体需要执行的任务
     * @param <T>             返回对象
     * @return FutureTask
     */
    public static <T> FutureTask<T> getFutureTask(ExecutorService executorService, Callable<T> callable) {
        FutureTask<T> task = new FutureTask<>(callable);
        executorService.submit(task);
        return task;
    }

    /**
     * 获取异步任务返回结果
     *
     * @param task    异步任务
     * @param timeout 超时时间
     * @param unit    时间单位
     * @param name    小类
     * @return 异步任务返回结果
     */
    public static <T> T getFutureTaskResult(FutureTask<T> task, long timeout, TimeUnit unit, String name) {
        try {
            if (task == null) {
                return null;
            }
            return task.get(timeout, unit);
        } catch (Exception e) {
            if (e instanceof TimeoutException) {
                if (StringUtils.isNotBlank(name)) {
                    //cat埋点，用于记录超时情况
                    BigdataCatUtil.logEvent("timeout", name);
                }
            } else if (e instanceof ExecutionException) {
                if (StringUtils.isNotBlank(name)) {
                    //cat埋点，用于记录超时情况
                    BigdataCatUtil.logEvent("executionException", name);
                }
            } else {
                log.error("getFutureTaskResult error, name = {} ", name, e);
            }
        }

        return null;
    }

    /**
     * 异步提交需要执行的任务，并等待返回结果
     *
     * @param callable              需要执行的任务
     * @param submitExecutorService 任务提交线程池
     * @param taskExecutorService   任务执行线程池
     * @param timeout               返回结果超时时间
     * @param name                  小类
     * @param <T>                   返回结果对象
     * @return FutureTask
     */
    public static <T> FutureTask<T> getFutureTask(Callable<T> callable, ExecutorService submitExecutorService,
                                                  ExecutorService taskExecutorService, long timeout, String name) {
        return getFutureTask(submitExecutorService,
                () -> getFutureTaskResult(getFutureTask(taskExecutorService, callable), timeout, TimeUnit.MILLISECONDS, name)
        );
    }

    /**
     * 异步提交需要执行的任务，并等待返回结果，ExecutorService.invokeAll实现
     *
     * @param callables           需要执行的任务
     * @param taskExecutorService 任务执行线程池
     * @param timeout             返回结果超时时间
     * @param <T>                 返回结果对象
     * @return
     */
    public static <T> List<Future<T>> getFutureList(List<Callable<T>> callables, ExecutorService taskExecutorService, long timeout) {

        try {
            return taskExecutorService.invokeAll(callables, timeout, TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            log.error("getFutureList with timeout error = ", e);
        }

        return null;
    }

    /**
     * 异步提交需要执行的任务，并等待返回结果，ExecutorService.invokeAll实现
     *
     * @param callables           需要执行的任务
     * @param taskExecutorService 任务执行线程池
     * @param <T>                 返回结果对象
     * @return
     */
    public static <T> List<Future<T>> getFutureList(List<Callable<T>> callables, ExecutorService taskExecutorService) {

        try {
            return taskExecutorService.invokeAll(callables);
        } catch (Exception e) {
            log.error("getFutureList error = ", e);
        }

        return null;
    }

    public static List<Object> getFutureResult(List<Future<Object>> futures){

        if(CollectionUtils.isEmpty(futures)){
            return Collections.emptyList();
        }

        List<Object> list = Lists.newArrayList();
        try{
            for (Future<Object> future : futures) {
                if(future.isCancelled()){
                    list.add(null);
                    continue;
                }

                list.add(future.get());
            }

        }catch (Exception e){
            log.error("getFutureResult error = ", e);
        }

        return list;
    }

}
