package cn.com.duiba.bigdata.common.biz.utils;

import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;

import java.io.IOException;

/**
 * @author xugf hdfs文件系统工具类
 * 需要引入 duiba-core-site.xml 和 duiba-hdfs-site.xml 配置文件
 */
public class HDFSUtil {
    //hdfs文件系统
    private FileSystem fs;

    public HDFSUtil() throws IOException {
        ClassLoader loader = HDFSUtil.class.getClassLoader();
        Configuration conf = new Configuration();
        conf.addResource(loader.getResource("duiba-core-site.xml"));
        conf.addResource(loader.getResource("duiba-hdfs-site.xml"));
        conf.set("fs.hdfs.impl", "org.apache.hadoop.hdfs.DistributedFileSystem");
        //支持往文件里追加数据
        conf.setBoolean("dfs.support.append", true);
        if (System.getProperties().getProperty("os.name").toLowerCase().contains("windows")) {
            conf.setBoolean("mapreduce.app-submission.cross-platform", true);
        }
        this.fs = FileSystem.get(conf);
    }

    //判断FileSystem连接是否已经关闭
    public boolean isAlive() {
        return fs != null;
    }

    /**
     * 关闭FileSystem连接
     */
    public void close() throws IOException {
        if (fs != null) {
            fs.close();
            fs = null;
        }
    }

    /**
     * 同步递归创建hdfs目录
     *
     * @param hdfsPath hdfs目录
     * @return true：目录创建成功 false：目录创建失败
     * @throws IOException IOException
     */
    public synchronized boolean mkdirs(String hdfsPath) throws IOException {
        Path path = new Path(hdfsPath);
        if (!fs.exists(path)) {
            if (!fs.mkdirs(path)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 重命名hdfs文件
     *
     * @param source 源文件名称
     * @param dest   重命名后的文件名称
     * @return true：重命名成功 false：重命名失败
     * @throws IOException IOException
     */
    public boolean rename(String source, String dest) throws IOException {
        //当文件已经存在时，删除
        if (fs.exists(new Path(dest))) {
            fs.delete(new Path(dest), false);
        }

        //重命名文件
        return fs.rename(new Path(source), new Path(dest));
    }

    /**
     * 获取hdfs文件大小
     *
     * @param file 文件名称
     * @return 文件大小
     * @throws IOException IOException
     */
    public long getFileLength(String file) throws IOException {
        long size = 0L;
        Path path = new Path(file);
        if (fs.exists(path)) {
            size = fs.getFileStatus(path).getLen();
        }
        return size;
    }

    /**
     * 创建文件输入流
     *
     * @param filePath 文件全路径
     * @return 输入流
     * @throws IOException IOException
     */
    public FSDataOutputStream createOutputStream(String filePath) throws IOException {
        FSDataOutputStream fos;
        Path path = new Path(filePath);
        if (fs.exists(path)) {
            fos = fs.append(path);
        } else {
            fos = fs.create(path);
        }
        //设置文件权限
        fs.setPermission(path, new FsPermission((short) 0777));
        return fos;
    }

    /**
     * @param filePath  文件全路径
     * @param recursive 是否递归删除，目录设置成true，文件可以设置true或者false
     * @return
     * @throws IOException
     */
    public synchronized boolean delete(String filePath, boolean recursive) throws IOException {

        if (StringUtils.isBlank(filePath)) {
            return false;
        }

        Path path = new Path(filePath);
        if (!fs.exists(path)) {
            return false;
        }

        return fs.delete(path, recursive);
    }

    /**
     * 判断文件是否存在
     *
     * @param filePath
     * @return
     * @throws IOException
     */
    public boolean fileExists(String filePath) throws IOException {
        Path path = new Path(filePath);
        return fs.exists(path);
    }


    /**
     * 创建文件输出流
     *
     * @param filePath 文件全路径
     * @return 输出流
     * @throws IOException IOException
     */
    public FSDataInputStream createInputStream(String filePath) throws IOException {
        Path path = new Path(filePath);
        if (!fs.exists(path)) {
            throw new IOException("文件不存在！filePath = " + filePath);
        }

        return fs.open(path);
    }
}
