package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.jena.sparql.expr.*;
import org.apache.jena.sparql.util.ExprUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 标签过滤组合转化工具类
 */
@Slf4j
public class LabelSetTransformUtil {

    public static final String QUESTION_MARK = "?";

    public static void main(String[] args) {

        //测试交集的情况
        Expr expr1 = ExprUtils.parse("(?010301 || ?010302 || ?010303 ) && ( ?010201 || ?010202 || ?010204)");

        Expr expr2 = ExprUtils.parse("(?010303 ) && ( ?010201 )");

        Expr expr3 = ExprUtils.parse("(?010303 && ?01040201 ) && ( ?010201 )");

        Expr expr4 = ExprUtils.parse("(?010303 || ?01040201 ) && ( ?010201 )");

        Expr expr5 = ExprUtils.parse("(?010303) && ( ?01040201 || ?010201 )");

        Expr expr6 = ExprUtils.parse("(?010301 || ?020302 || ?030303 ) && ( ?040201 || ?050202 || ?060204)");


        //测试并集的情况
        Expr expr7 = ExprUtils.parse("(?010301) || ( ?010201)");

        Expr expr8 = ExprUtils.parse("(?010301) || ( ?010302)");

        Expr expr9 = ExprUtils.parse("(?010301 || ?010303) || ( ?020302)");

        //交并混合
        Expr expr10 = ExprUtils.parse("(?020210001 || ?031102) && ( ?010101)");

        Expr expr11 = ExprUtils.parse("(?020210001 && ?031102) || ( ?010101)");

        Expr expr12 = ExprUtils.parse("(?020210001 || ?031102) || ( ?010101)");

        Expr expr13 = ExprUtils.parse("(?020210001 && ?010101) || ( ?020210002 && ?010102)|| ?031102");

        //应用兴趣类标签组合
        Expr expr14 = ExprUtils.parse("(?031001 && ?031105)");

        Expr expr15 = ExprUtils.parse("(?031001 || ?031105 || ?031201 ) && ( ?040201 || ?050202 || ?060204)");

        //排除的情况
        Expr expr16 = ExprUtils.parse("!(?031001 && ?031105 && ?041105)");

        Expr expr17 = ExprUtils.parse("(?020210001 && ?031102) || !(?031001 || ?031105)");

        Expr expr18 = ExprUtils.parse("(?020210001 || ?031102) && !(?031001 || ?031105)");

        Expr expr19 = ExprUtils.parse("((?0108||?010401||?010402)||(?010501||?010502))&&!(?010904||?010905||?010906||?010903||?010902)");

        Expr expr = ExprUtils.parse("(?0108)");


        DistributeLabelFilterRuleDTO distributeLabelFilterRuleDTO = distributiveLabelFilterRule(expr.toString());
        System.out.println(JSONObject.toJSONString(distributeLabelFilterRuleDTO));
    }

    /**
     * 目前给DMP后端使用
     * 将标签组合过滤规则拆解，拆成若干个只有交集的并的组合
     * 需要拆解的标签过滤规则，如：labelFilterRule ：(?0701 || ?0702 || ?0703 ) && ( ?010201 || ?010202 || ?010205)
     * 拆解后的结果：不拆
     * 需要拆解的标签过滤规则，如：labelFilterRule ：(?020210202  && 010101 ) || ( ?020210201 && ?010102 )
     * 拆解后的结果：(?020210202  && 010101 ),( ?020210201 && ?010102 )
     * @param labelFilterRule
     * @return
     */
    public static DistributeLabelFilterRuleDTO distributiveLabelFilterRule(String labelFilterRule){

        if (StringUtils.isEmpty(labelFilterRule) || !labelFilterRule.contains(QUESTION_MARK)) {
            return null;
        }

        try{
            Expr expr = ExprUtils.parse(labelFilterRule);
            if(expr instanceof E_LogicalAnd){
               return fetchDistributeLabelFilterRuleForAnd((E_LogicalAnd) expr);
            }else if (expr instanceof E_LogicalOr){
               return fetchDistributeLabelFilterRuleForOr((E_LogicalOr) expr);
            } else if(expr instanceof E_LogicalNot){
                return fetchDistributeLabelFilterRuleForNot((E_LogicalNot) expr);
            }else if(expr instanceof ExprVar){
                DistributeLabelFilterRuleDTO distributeLabelFilterRuleDTO = new DistributeLabelFilterRuleDTO();
                distributeLabelFilterRuleDTO.setNeedSplit(Boolean.FALSE);
                return distributeLabelFilterRuleDTO;
            }
        }catch (Exception e){

            return null;
        }

        return null;
    }

    /**
     * 目前给大数据过滤服务使用
     * 获取标签组合的所有标签元素
     *
     * @param labelCompose 标签组合
     * @return
     */
    public static List<String> fetchLabelComposeElements(String labelCompose) {

        if (StringUtils.isEmpty(labelCompose) || !labelCompose.contains(QUESTION_MARK)) {
            return null;
        }

        try {
            List<String> elements = Lists.newArrayList();
            Expr expr = ExprUtils.parse(labelCompose);
            fetchElementFromLogical(expr, elements);
            return elements;
        } catch (Exception e) {
            log.warn("LabelSetTransformUtil.fetchLabelComposeElements labelCompose = {} error ={} ", labelCompose, e);
        }

        return null;
    }

    private static  DistributeLabelFilterRuleDTO fetchDistributeLabelFilterRuleForNot(E_LogicalNot expr) {

        DistributeLabelFilterRuleDTO distributeLabelFilterRuleDTO = new DistributeLabelFilterRuleDTO();
        E_LogicalNot e_logicalNot = expr;
        Expr arg = e_logicalNot.getArg();
        StringBuilder sb = new StringBuilder();

        List<String> elements = Lists.newArrayList();
        fetchElementFromLogical(expr,elements);
        if(arg instanceof E_LogicalAnd){
            for (String element : elements) {
                sb.append("!( ?").append(element).append(" )").append(" || ");
            }
        }else if(arg instanceof E_LogicalOr){
            for (String element : elements) {
                sb.append("!( ?").append(element).append(" )").append(" && ");
            }

        }

        String s = sb.toString();
        distributeLabelFilterRuleDTO.setFilters(Lists.newArrayList(s.substring(0,s.length()-4)));
        distributeLabelFilterRuleDTO.setNeedSplit(Boolean.TRUE);

        return distributeLabelFilterRuleDTO;
    }


    private static DistributeLabelFilterRuleDTO fetchDistributeLabelFilterRuleForOr(E_LogicalOr expr) {

        DistributeLabelFilterRuleDTO distributeLabelFilterRuleDTO = new DistributeLabelFilterRuleDTO();
        E_LogicalOr e_logicalOr = expr;

        List<String> elements = Lists.newArrayList();

        fetchClassElementFromLogical(e_logicalOr,elements);
        int leftNumOr = e_logicalOr.toString().split("\\|\\|").length;

        boolean notNeedDivide = false;
        if(elements.size()==leftNumOr){
            notNeedDivide = true;
        }else{
            Expr arg1 = e_logicalOr.getArg1();
            Expr arg2 = e_logicalOr.getArg2();

            if(arg1 instanceof ExprVar && arg2 instanceof  ExprVar){
                notNeedDivide = getLabelClass(arg1.getVarName()).equalsIgnoreCase(getLabelClass(arg2.getVarName()));
            }else if (arg1 instanceof E_LogicalOr || arg2 instanceof E_LogicalOr){
                List<String> leftElements = Lists.newArrayList();
                fetchClassElementFromLogical(arg1,leftElements);

                boolean leftSameClassElemnt = leftElements.stream().distinct().count()==1;
                int leftSubNumOr = arg1.toString().split("\\|\\|").length;
                if(leftSameClassElemnt&&leftElements.size()==leftSubNumOr){
                    notNeedDivide = false;
                }

                List<String> rightElements = Lists.newArrayList();
                fetchClassElementFromLogical(arg2,rightElements);

                boolean rightSameClassElemnt = rightElements.stream().distinct().count()==1;
                int rightSubNumOr = arg1.toString().split("\\|\\|").length;
                if(rightSameClassElemnt&&rightElements.size()==rightSubNumOr){
                    notNeedDivide = false;
                }
            }else if (arg1 instanceof E_LogicalNot || arg2 instanceof E_LogicalNot){
                notNeedDivide = false;
            }
        }

        if(!notNeedDivide){
            List<String> filterRule = distributiveFilterRule(expr);
            distributeLabelFilterRuleDTO.setFilters(filterRule);
            distributeLabelFilterRuleDTO.setNeedSplit(Boolean.TRUE);
        }else{
            distributeLabelFilterRuleDTO.setNeedSplit(Boolean.FALSE);
        }

        return distributeLabelFilterRuleDTO;
    }

    /**
     * 原始过滤规则或逻辑处理
     * @param expr
     * @return
     */
    private static DistributeLabelFilterRuleDTO fetchDistributeLabelFilterRuleForAnd(E_LogicalAnd expr) {

        DistributeLabelFilterRuleDTO distributeLabelFilterRuleDTO = new DistributeLabelFilterRuleDTO();
        E_LogicalAnd e_logicalAnd = expr;

        Expr arg1 = e_logicalAnd.getArg1();
        List<String> leftElements = Lists.newArrayList();

        int leftNumOr = 0;
        if(arg1 instanceof  E_LogicalOr){
            fetchClassElementFromLogical(arg1,leftElements);
            leftNumOr = arg1.toString().split("\\|\\|").length;
        }else if(arg1 instanceof ExprVar){
            leftElements.add(getLabelClass(arg1.getVarName()));
            leftNumOr =1;
        }

        boolean leftClassElement = true;
        if(CollectionUtils.isNotEmpty(leftElements)){
            leftClassElement = leftElements.stream().distinct().count() == 1;
        }

        Expr arg2 = e_logicalAnd.getArg2();
        List<String> rightElements = Lists.newArrayList();
        int rightNumOr = 0;

        if(arg2 instanceof E_LogicalOr){
            fetchClassElementFromLogical(arg2,rightElements);
            rightNumOr = arg2.toString().split("\\|\\|").length;
        }else if(arg2 instanceof ExprVar){
            rightElements.add(getLabelClass(arg2.getVarName()));
            rightNumOr =1;
        }

        boolean rightClassElement = true;
        if(CollectionUtils.isNotEmpty(rightElements)){
            rightClassElement = rightElements.stream().distinct().count() == 1;
        }

        boolean notNeedDivide = leftClassElement&&leftElements.size()==leftNumOr && rightElements.size()==rightNumOr&&rightClassElement;
        if(!notNeedDivide){
            List<String> filterRule = distributiveFilterRule(expr);
            distributeLabelFilterRuleDTO.setFilters(filterRule);
            distributeLabelFilterRuleDTO.setNeedSplit(Boolean.TRUE);
        }else{
            distributeLabelFilterRuleDTO.setNeedSplit(Boolean.FALSE);
        }

        return distributeLabelFilterRuleDTO;
    }


    public static List<String> distributiveFilterRule(Expr expr) {

        List<String> labelSet = Lists.newArrayList();
        try {

            if (expr instanceof E_LogicalAnd) {
                E_LogicalAnd e_logicalAnd = (E_LogicalAnd) expr;
                String opName = e_logicalAnd.getOpName();

                List<String> left = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalAnd.getArg1(), left);

                List<String> right = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalAnd.getArg2(), right);

                for (String l : left) {
                    for (String r : right) {
                        StringBuilder sb = new StringBuilder();
                        labelSet.add(sb.append("(").append(l).append(" ").append(opName).append(" ").append(r).append(")").toString());
                    }
                }

            } else if (expr instanceof E_LogicalOr) {
                E_LogicalOr e_logicalOr = (E_LogicalOr) expr;

                List<String> left = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalOr.getArg1(), left);

                List<String> right = Lists.newArrayList();
                concatStrForLogicalOr(e_logicalOr.getArg2(), right);

                labelSet.addAll(left);
                labelSet.addAll(right);
            }
        } catch (Exception e) {
            log.warn("LabelSetTransformUtil.distributiveFilterRule labelFilterRule = {} error = {}", expr.toString(), e);
        }

        return labelSet;
    }


    /**
     * 针对逻辑或的并操作进行解析标签组合过滤规则，递归处理
     *
     * @param expr
     * @param param
     * @return
     */
    public static String concatStrForLogicalOr(Expr expr, List<String> param) {
        if (expr instanceof ExprVar) {
            ExprVar ev = (ExprVar) expr;
            param.add(ev.toString());
        } else if (expr instanceof E_LogicalOr) {
            E_LogicalOr ssub1 = (E_LogicalOr) expr;

            List<String> elements = Lists.newArrayList();
            fetchClassElementFromLogical(ssub1,elements);

            boolean leftSameClassElemnt = elements.stream().distinct().count()==1;
            int leftSubNumOr = ssub1.toString().split("\\|\\|").length;
            if(leftSameClassElemnt&&elements.size()==leftSubNumOr){
                param.add(ssub1.toString());
            }else{
                String s1 = concatStrForLogicalOr(ssub1.getArg1(), param);
                if (StringUtils.isNotEmpty(s1)) {
                    param.add(s1);
                }

                String s2 = concatStrForLogicalOr(ssub1.getArg2(), param);
                if (StringUtils.isNotEmpty(s2)) {
                    param.add(s2);
                }
            }

        } else if (expr instanceof E_LogicalAnd) {
            param.add(expr.toString());
        }else if (expr instanceof E_LogicalNot){
            E_LogicalNot e_logicalNot = (E_LogicalNot) expr;
            Expr arg = e_logicalNot.getArg();
            StringBuilder sb = new StringBuilder();

            List<String> elements = Lists.newArrayList();
            fetchElementFromLogical(expr,elements);
            if(arg instanceof E_LogicalAnd){
                for (String element : elements) {
                    sb.append("!( ?").append(element).append(" )").append(" || ");
                }
            }else if(arg instanceof E_LogicalOr){
                for (String element : elements) {
                    sb.append("!( ?").append(element).append(" )").append(" && ");
                }
            }
            String s = sb.toString();
            param.add(s.substring(0,s.length()-4));
        }

        return null;
    }

    /**
     * 针对逻辑或的并操作提取元素，递归处理
     *
     * @param expr
     * @param elements
     * @return
     */
    public static String fetchElementFromLogical(Expr expr, List<String> elements) {
        if (expr instanceof ExprVar) {
            ExprVar ev = (ExprVar) expr;
            elements.add(ev.getVarName());
        } else if (expr instanceof E_LogicalOr) {
            E_LogicalOr e_logicalOr = (E_LogicalOr) expr;
            String s1 = fetchElementFromLogical(e_logicalOr.getArg1(), elements);
            if (StringUtils.isNotEmpty(s1)) {
                elements.add(s1);
            }

            String s2 = fetchElementFromLogical(e_logicalOr.getArg2(), elements);
            if (StringUtils.isNotEmpty(s2)) {
                elements.add(s2);
            }

        } else if (expr instanceof E_LogicalAnd) {
            E_LogicalAnd e_logicalAnd = (E_LogicalAnd) expr;
            String s1 = fetchElementFromLogical(e_logicalAnd.getArg1(), elements);
            if (StringUtils.isNotEmpty(s1)) {
                elements.add(s1);
            }

            String s2 = fetchElementFromLogical(e_logicalAnd.getArg2(), elements);
            if (StringUtils.isNotEmpty(s2)) {
                elements.add(s2);
            }
        }else if(expr instanceof E_LogicalNot){
            E_LogicalNot e_logicalNot = (E_LogicalNot) expr;
            List<Expr> args = e_logicalNot.getArgs();
            for (Expr arg : args) {
                fetchElementFromLogical(arg,elements);
            }

        }

        return null;
    }


    private static String getLabelClass(String labelCode){
        if(StringUtils.isEmpty(labelCode)){
            return null;
        }

        if(labelCode.startsWith("03")){
            return "0300";
        }else if(labelCode.startsWith("04")){
            return "0400";
        }else if(labelCode.startsWith("05")){
            return "0500";
        }else if(labelCode.startsWith("09")){
            return "0900";
        }

        return labelCode.substring(0,4);
    }

    /**
     * 针对逻辑或的并操作提取标签类别，递归处理
     *
     * @param expr
     * @param elements
     * @return
     */
    public static String fetchClassElementFromLogical(Expr expr, List<String> elements) {
        if (expr instanceof ExprVar) {
            ExprVar ev = (ExprVar) expr;
            elements.add(getLabelClass(ev.getVarName()));
        } else if (expr instanceof E_LogicalOr) {
            E_LogicalOr e_logicalOr = (E_LogicalOr) expr;
            String s1 = fetchClassElementFromLogical(e_logicalOr.getArg1(), elements);
            if (StringUtils.isNotEmpty(s1)) {
                elements.add(getLabelClass(s1));
            }

            String s2 = fetchClassElementFromLogical(e_logicalOr.getArg2(), elements);
            if (StringUtils.isNotEmpty(s2)) {
                elements.add(getLabelClass(s2));
            }

        } else if (expr instanceof E_LogicalAnd) {
            E_LogicalAnd e_logicalAnd = (E_LogicalAnd)expr;
            String s1 = fetchClassElementFromLogical(e_logicalAnd.getArg1(), elements);
            if (StringUtils.isNotEmpty(s1)) {
                elements.add(getLabelClass(s1));
            }

            String s2 = fetchClassElementFromLogical(e_logicalAnd.getArg2(), elements);
            if (StringUtils.isNotEmpty(s2)) {
                elements.add(getLabelClass(s2));
            }
        }

        return null;
    }

    @Data
    public static class DistributeLabelFilterRuleDTO {
        private List<String> filters;

        private Boolean needSplit;


    }


}
