package cn.com.duiba.bigdata.common.biz.utils;

import com.aliyun.oss.OSS;
import com.aliyun.oss.OSSClientBuilder;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.OSSObjectSummary;
import com.aliyun.oss.model.ObjectListing;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

/**
 * 阿里云oss 工具类，简化开发接入
 */
public class AliOssUtil {
    //oss客户端
    private OSS ossClient = null;

    //填写Bucket所在地域对应的Endpoint
    private String endpoint;

    //阿里云ram账号秘钥
    private String accessKeyId;
    private String accessKeySecret;

    public void setEndpoint(String endpoint) {
        this.endpoint = endpoint;
    }

    public void setAccessKeyId(String accessKeyId) {
        this.accessKeyId = accessKeyId;
    }

    public void setAccessKeySecret(String accessKeySecret) {
        this.accessKeySecret = accessKeySecret;
    }

    /**
     * 获取oss客户端
     *
     * @return oss客户端
     */
    protected OSS getOssClient() {
        if (ossClient != null) {
            return ossClient;
        }

        return initOssClient();
    }

    /**
     * 初始化oss连接
     *
     * @return oss客户端
     */
    private synchronized OSS initOssClient() {
        if (ossClient != null) {
            return ossClient;
        }

        ossClient = new OSSClientBuilder().build(endpoint, accessKeyId, accessKeySecret);

        return ossClient;
    }

    /**
     * 关闭oss连接
     */
    public void close() {
        ossClient.shutdown();
    }

    /**
     * 上传数据到oss中，如果文件已经存在，会覆盖写
     *
     * @param bucketName bucket名称
     * @param objectName 文件所在的路径
     * @param input      io流
     */
    public void putObject(String bucketName, String objectName, InputStream input) {
        getOssClient().putObject(bucketName, objectName, input);
    }

    /**
     * 获取文件流（注意点：数据读取完成后，获取的流必须关闭，否则会造成连接泄漏，导致请求无连接可用）
     *
     * @param bucketName bucket名称
     * @param objectName 文件所在的路径
     * @return io流
     */
    public InputStream getObjectContentInputStream(String bucketName, String objectName) {
        OSSObject ossObject = getOssClient().getObject(bucketName, objectName);

        // 调用ossObject.getObjectContent获取文件输入流，可读取此输入流获取其内容。
        return ossObject.getObjectContent();
    }

    /**
     * 列举出所有的文件
     * @param bucketName bucket名称
     * @param prefix 文件前缀
     * @return 文件名称列表
     */
    public List<String> listObjects(String bucketName, String prefix) {
        List<String> list = new ArrayList<>();
        ObjectListing objectListing = getOssClient().listObjects(bucketName, prefix);
        for (OSSObjectSummary ossObject : objectListing.getObjectSummaries()) {
            list.add(ossObject.getKey());
        }

        return list;
    }

    /**
     * 文件重命名
     * @param bucketName bucket名称
     * @param sourceObjectName 原文件名（包含路径）
     * @param destinationObjectName 新文件名（包含路径）
     */
    public void renameObject(String bucketName, String sourceObjectName, String destinationObjectName) {
        //拷贝文件
        getOssClient().copyObject(bucketName, sourceObjectName, bucketName, destinationObjectName);
        //删除原文件
        getOssClient().deleteObject(bucketName, sourceObjectName);
    }


}
