package cn.com.duiba.bigdata.common.biz.utils;

import com.alibaba.fastjson.JSON;
import com.google.common.base.CaseFormat;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.Cell;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.util.Bytes;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class HbaseResultUtil {
    private static final Logger logger = LoggerFactory.getLogger(HbaseResultUtil.class);


    //java 小驼峰格式
    public static String getLowerCamelColumn(boolean lowerCamel, String column) {
        if (lowerCamel && column.contains("_")) {
            return CaseFormat.LOWER_UNDERSCORE.to(CaseFormat.LOWER_CAMEL, column);
        }
        return column;
    }

    //组装单行数据
    public static void putMapData(Map<String, String> map, String column, String value, boolean lowerCamel) {
        if (StringUtils.isAnyBlank(column, value)) {
            return;
        }
        column = getLowerCamelColumn(lowerCamel, column);
        map.put(column, value);
    }

    //组装多行数据
    public static void putMapData(Map<String, Map<String, String>> map, String rowKey, Map<String, String> columnMap) {
        if (StringUtils.isBlank(rowKey)) {
            return;
        }
        if (columnMap == null || columnMap.size() == 0) {
            return;
        }
        map.put(rowKey, columnMap);
    }

    //解析单行单列
    public static String getResultString(Result result, String family, String column) {
        String value = "";
        if (result != null && result.containsColumn(family.getBytes(), column.getBytes())) {
            value = Bytes.toString(result.getValue(family.getBytes(), column.getBytes()));
        }
        return value;
    }

    //解析单行多列
    public static Map<String, String> getResultMap(Result result, String family, List<String> columns) {
        return getResultMap(result, family, columns, false);
    }

    //解析单行多列
    public static Map<String, String> getResultMap(Result result, String family, List<String> columns, boolean lowerCamel) {
        Map<String, String> map = new HashMap<>();
        for (String column : columns) {
            String value = getResultString(result, family, column);
            putMapData(map, column, value, lowerCamel);
        }
        return map;
    }

    //解析单行多列
    public static <T> T getResultMap(Result result, String family, List<String> columns, Class<T> clazz) {
        Map<String, String> map = getResultMap(result, family, columns, true);
        return JSON.parseObject(JSON.toJSONString(map), clazz);
    }

    //解析单行所有列
    public static Map<String, String> getResultMap(Result result, String family) {
        return getResultMap(result, family, false);
    }

    //解析单行所有列
    public static Map<String, String> getResultMap(Result result, String family, boolean lowerCamel) {
        if (result == null || result.isEmpty()) {
            return null;
        }
        List<Cell> cellList = result.listCells();
        if (CollectionUtils.isEmpty(cellList)) {
            return null;
        }
        Map<String, String> map = new HashMap<>();
        for (Cell cell : cellList) {
            if (!Bytes.toString(CellUtil.cloneFamily(cell)).equalsIgnoreCase(family)) {
                continue;
            }
            String column = Bytes.toString(CellUtil.cloneQualifier(cell));
            String value = Bytes.toString(CellUtil.cloneValue(cell));
            putMapData(map, column, value, lowerCamel);
        }
        return map;
    }

    //解析单行所有列
    public static <T> T getResultMap(Result result, String family, Class<T> clazz) {
        Map<String, String> map = getResultMap(result, family, true);
        return JSON.parseObject(JSON.toJSONString(map), clazz);
    }

    //解析多行单列
    public static Map<String, String> getResultMap(Result[] results, String family, String column) {
        Map<String, String> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            String value = getResultString(result, family, column);
            if (StringUtils.isNotBlank(value)) {
                map.put(rowKey, value);
            }
        }
        return map;
    }

    //解析多行多列
    public static Map<String, Map<String, String>> getResultMap(Result[] results, String family, List<String> columns) {
        Map<String, Map<String, String>> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            Map<String, String> columnMap = getResultMap(result, family, columns);
            putMapData(map, rowKey, columnMap);
        }
        return map;
    }

    //解析多行多列
    public static <T> Map<String, T> getResultMap(Result[] results, String family, List<String> columns, Class<T> clazz) {
        Map<String, T> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            T obj = getResultMap(result, family, columns, clazz);
            if (obj != null) {
                map.put(rowKey, obj);
            }
        }
        return map;
    }

    //解析多行多列
    public static Map<String, Map<String, String>> getResultMap(Result[] results, String family, Map<String, List<String>> rowkeyMap) {
        Map<String, Map<String, String>> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            Map<String, String> columnMap = getResultMap(result, family, rowkeyMap.get(rowKey));
            putMapData(map, rowKey, columnMap);
        }
        return map;
    }

    //解析多行多列
    public static <T> Map<String, T> getResultMap(Result[] results, String family, Map<String, List<String>> rowkeyMap, Class<T> clazz) {
        Map<String, T> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            T obj = getResultMap(result, family, rowkeyMap.get(rowKey), clazz);
            if (obj != null) {
                map.put(rowKey, obj);
            }
        }
        return map;
    }

    //解析多行所有列
    public static Map<String, Map<String, String>> getResultMap(Result[] results, String family) {
        Map<String, Map<String, String>> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            Map<String, String> columnMap = getResultMap(result, family);
            putMapData(map, rowKey, columnMap);
        }
        return map;
    }

    //解析多行所有列
    public static <T> Map<String, T> getResultMap(Result[] results, String family, Class<T> clazz) {
        Map<String, T> map = new HashMap<>();
        for (Result result : results) {
            if (result == null || result.isEmpty()) {
                continue;
            }
            String rowKey = Bytes.toString(result.getRow());
            T obj = getResultMap(result, family, clazz);
            if (obj != null) {
                map.put(rowKey, obj);
            }
        }
        return map;
    }

}
