package cn.com.duiba.bigdata.common.biz.service;

import cn.com.duiba.bigdata.common.biz.dto.KeyValueDto;
import cn.com.duiba.bigdata.common.biz.dto.SortDto;
import cn.com.duiba.bigdata.common.biz.enums.HologresShowTypeEnum;
import cn.com.duiba.bigdata.common.biz.enums.OperatorsEnum;
import cn.com.duiba.bigdata.common.biz.enums.SortEnum;
import cn.com.duiba.bigdata.common.biz.form.HoloQueryForm;
import cn.com.duiba.bigdata.common.biz.interfaces.CalculateMetricEnum;
import cn.com.duiba.bigdata.common.biz.interfaces.CombineMetricEnum;
import cn.com.duiba.bigdata.common.biz.interfaces.DimensionEnum;
import cn.com.duiba.bigdata.common.biz.interfaces.MetricEnum;
import cn.com.duiba.bigdata.common.biz.utils.BusinessEnumUtil;
import cn.com.duiba.bigdata.common.biz.utils.MD5Util;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.*;

/**
 * @author xugf 2023-11-28
 * 通过查询条件，自动组装查询sql
 */
@Slf4j
public class HoloSqlService {

    /**
     * 组装查询sql语句
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     * @return sql
     * @throws Exception 异常
     */
    public static String getQuerySql(HoloQueryForm form, int businessType) throws Exception {

        //AB实验条件预处理
        dealABTestCondition(form);

        //添加排序指标
        addOrderMetricList(form, businessType);

        //添加计算指标对应的基础指标
        addCalculateMetricList(form, businessType);

        //添加组合指标对应的基础指标
        addCombineMetricList(form, businessType);

        //对查询指标进行分组合并
        List<Object> metricList = getMetricList(form.getMetricList(), businessType);

        //获取指标条件
        List<KeyValueDto> metricConditionList = getMetricConditionList(form);

        //组装多层嵌套sql语句
        StringBuilder sql = new StringBuilder("select ");

        //查询维度
        String dimensionSumSql = getDimensionSumSql(form.getDataShowType(), form.getDimensionList(), businessType);
        if (StringUtils.isNotBlank(dimensionSumSql)) {
            sql.append(dimensionSumSql);
        }

        //查询指标
        String metricSumSql = getMetricSumSql(metricList, form.getCalculateMetricList(), form.getCombineMetricList(), businessType);
        //指标一定会有值
        sql.append(metricSumSql);

        //子查询
        String querySql = getQueryUnionAllSql(form, metricList, businessType);
        sql.append(" from (").append(querySql).append(") t ");

        //group by
        String groupBySumSql = getGroupBySumSql(form.getDataShowType(), form.getDimensionList(), businessType);
        if (StringUtils.isNotBlank(groupBySumSql)) {
            sql.append(" group by ").append(groupBySumSql);
        }

        //having 条件
        String havingSql = getHavingSumSql(metricConditionList, businessType);
        if (StringUtils.isNotBlank(havingSql)) {
            sql.append(" having ").append(havingSql);
        }

        //order by
        String orderBySql = getOrderBySql(form.getDataShowType(), form.getOrderByMetric(), businessType);
        if (StringUtils.isNotBlank(orderBySql)) {
            sql.append(orderBySql);
        }

        //分页展示 limit 5 offset 10 ;
        if (form.getPageSize() != null && form.getPageNumber() != null) {
            sql.append(" limit ").append(form.getPageSize()).append(" offset ").append((form.getPageNumber() - 1) * form.getPageSize());
        }

        return sql.toString();
    }

    /**
     * AB实验条件预处理
     *
     * @param form 查询条件
     */
    private static void dealABTestCondition(HoloQueryForm form) {
        if (CollectionUtils.isEmpty(form.getConditionList())) {
            return;
        }

        //实验计划id
        String abtestPlanId = "";
        //实验分组id
        String abtestGroupId = "";
        //条件列表
        List<KeyValueDto> conditionList = new ArrayList<>();

        for (KeyValueDto dto : form.getConditionList()) {
            if (dto.getKey().toString().equals(SqlService.ABTEST_PLAN_ID)) {
                abtestPlanId = dto.getValue().toString();
            } else if (dto.getKey().toString().equals(SqlService.ABTEST_GROUP_ID)) {
                abtestGroupId = dto.getValue().toString();
            } else {
                conditionList.add(dto);
            }
        }

        if (StringUtils.isNoneBlank(abtestPlanId, abtestGroupId)) {
            String abtest = abtestPlanId + "-" + abtestGroupId;
            //新增实验计划数组过滤条件
            conditionList.add(new KeyValueDto(SqlService.ABTEST, abtest, OperatorsEnum.ARRAY.toString()));

            //替换掉原有的查询条件
            form.setConditionList(conditionList);
        }
    }

    /**
     * 添加排序指标
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     */
    private static void addOrderMetricList(HoloQueryForm form, int businessType) {
        if (form.getOrderByMetric() == null) {
            return;
        }

        String metric = form.getOrderByMetric().getKey();
        if (Objects.requireNonNull(BusinessEnumUtil.getMetricEnumList(businessType)).contains(metric)) {
            List<String> metricList = form.getMetricList();
            if (metricList == null) {
                metricList = new ArrayList<>();
                form.setMetricList(metricList);
            }
            metricList.add(metric);
        } else if (Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnumList(businessType)).contains(metric)) {
            List<String> metricList = form.getCalculateMetricList();
            if (metricList == null) {
                metricList = new ArrayList<>();
                form.setCalculateMetricList(metricList);
            }
            metricList.add(metric);
        } else if (Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnumList(businessType)).contains(metric)) {
            List<String> metricList = form.getCombineMetricList();
            if (metricList == null) {
                metricList = new ArrayList<>();
                form.setCombineMetricList(metricList);
            }
            metricList.add(metric);
        }

    }

    /**
     * 将计算类指标对应的分子和分母指标加入到基础指标列表中
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     */
    private static void addCalculateMetricList(HoloQueryForm form, int businessType) {
        if (CollectionUtils.isEmpty(form.getCalculateMetricList())) {
            return;
        }

        if (CollectionUtils.isEmpty(form.getMetricList())) {
            //初始化基础指标列表
            form.setMetricList(new ArrayList<>());
        }

        //将计算类指标对应的分子和分母指标加入到基础指标列表中
        for (String metric : form.getCalculateMetricList()) {
            CalculateMetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, metric));
            form.getMetricList().add(metricEnum.getNumerator());
            form.getMetricList().add(metricEnum.getDenominator());
        }

    }

    /**
     * 将组合类指标加入到基础指标列表中
     *
     * @param form         查询条件
     * @param businessType 业务线类型
     */
    private static void addCombineMetricList(HoloQueryForm form, int businessType) {
        if (CollectionUtils.isEmpty(form.getCombineMetricList())) {
            return;
        }

        if (CollectionUtils.isEmpty(form.getMetricList())) {
            //初始化基础指标列表
            form.setMetricList(new ArrayList<>());
        }

        //将组合类指标加入到基础指标列表中
        for (String combineMetric : form.getCombineMetricList()) {
            CombineMetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, combineMetric));
            String[] array = StringUtils.split(metricEnum.getMetricList(), ",");
            for (String metric : array) {
                form.getMetricList().add(metric);
            }
        }

    }

    /**
     * 对查询指标进行分组合并
     *
     * @param metricList   查询的指标列表
     * @param businessType 业务线类型
     * @return 分组后的指标列表
     */
    private static List<Object> getMetricList(List<String> metricList, int businessType) {
        //去除重复的指标
        Set<String> metricSet = new HashSet<>(metricList);

        //分组指标
        List<Object> list = new ArrayList<>();

        //指标聚合map
        Map<String, List<String>> map = new HashMap<>();
        for (String metric : metricSet) {
            MetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metric));
            StringBuilder sb = new StringBuilder();
            //指标对应的 hologres 表
            sb.append(metricEnum.getTableName());
            //指标特定的查询条件
            sb.append(metricEnum.getCondition());

            String md5 = MD5Util.computeMD5(sb.toString());

            //map指标分组初始化
            if (!map.containsKey(md5)) {
                map.put(md5, new ArrayList<>());
            }

            //将指标放入map分组中
            map.get(md5).add(metric);
        }

        //将分组后的指标进行返回
        for (List<String> metricGroupList : map.values()) {
            if (metricGroupList.size() == 1) {
                list.add(metricGroupList.get(0));
            } else {
                list.add(metricGroupList);
            }
        }

        return list;
    }

    /**
     * 获取指标条件
     *
     * @param form 查询条件
     * @return 指标条件列表
     */
    private static List<KeyValueDto> getMetricConditionList(HoloQueryForm form) {
        if (CollectionUtils.isEmpty(form.getConditionList())) {
            return null;
        }

        //维度条件
        List<KeyValueDto> dimConditionList = new ArrayList<>();

        //指标条件
        List<KeyValueDto> metricConditionList = new ArrayList<>();

        for (KeyValueDto dto : form.getConditionList()) {
            int type = OperatorsEnum.valueOf(dto.getSymbol()).getType();
            if (type == 1) {
                dimConditionList.add(dto);
            } else if (type == 2) {
                metricConditionList.add(dto);
            }
        }

        if (metricConditionList.size() > 0) {
            //重置维度条件，去除指标条件
            form.setConditionList(dimConditionList);
        }

        return metricConditionList;
    }

    /**
     * 维度列表对应的sql查询片段
     *
     * @param dimensionList 查询的维度列表
     * @param businessType  业务线类型
     * @return sql查询片段
     */
    private static String getDimensionSumSql(String dataShowType, List<String> dimensionList, int businessType) {
        StringBuilder sb = new StringBuilder();

        //数据展示格式
        if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
            HologresShowTypeEnum hologresShowTypeEnum = HologresShowTypeEnum.valueOf(dataShowType);
            sb.append(hologresShowTypeEnum.getFormatSegment()).append(" as ").append(hologresShowTypeEnum.getFieldName()).append(", ");
        }

        sb.append(getDimensionSumSql(dimensionList, businessType));

        return sb.toString();
    }

    /**
     * 维度列表对应的sql查询片段
     *
     * @param dimensionList 查询的维度列表
     * @param businessType  业务线类型
     * @return sql查询片段
     */
    private static String getDimensionSumSql(List<String> dimensionList, int businessType) {
        StringBuilder sb = new StringBuilder();

        if (CollectionUtils.isEmpty(dimensionList)) {
            return sb.toString();
        }

        for (String dim : dimensionList) {
            DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dim));
            sb.append(dimensionEnum.getResultFieldName()).append(", ");
        }

        return sb.toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metricList          指标列表
     * @param calculateMetricList 计算类指标列表
     * @param businessType        业务线类型
     * @return 指标sql片段
     */
    private static String getMetricSumSql(List<Object> metricList, List<String> calculateMetricList, List<String> combineMetricList, int businessType) {
        StringBuilder sb = new StringBuilder();

        //基础指标
        for (Object metric : metricList) {
            sb.append(getObjectMetricSumSql(metric, businessType));
        }

        //计算指标
        if (CollectionUtils.isEmpty(calculateMetricList)) {
            return sb.deleteCharAt(sb.length() - 1).toString();
        }

        //去除重复的指标
        Set<String> calculateMetricSet = new HashSet<>(calculateMetricList);
        for (String metric : calculateMetricSet) {
            sb.append(getCalculateMetricSumSql(metric, businessType));
        }

        //组合指标
        if (CollectionUtils.isEmpty(combineMetricList)) {
            return sb.deleteCharAt(sb.length() - 1).toString();
        }

        //去除重复的指标
        Set<String> combineMetricSet = new HashSet<>(combineMetricList);
        for (String metric : combineMetricSet) {
            sb.append(getCombineMetricSumSql(metric, businessType));
        }

        return sb.deleteCharAt(sb.length() - 1).toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metric       指标
     * @param businessType 业务线类型
     * @return 指标sql片段
     */
    private static String getObjectMetricSumSql(Object metric, int businessType) {
        StringBuilder sb = new StringBuilder();

        if (metric instanceof String) {
            sb.append(getStringMetricSumSql(metric.toString(), businessType));
        } else if (metric instanceof List) {
            List<String> metricGroup = (List<String>) metric;
            for (String subMetric : metricGroup) {
                sb.append(getStringMetricSumSql(subMetric, businessType));
            }
        }

        return sb.toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metric       指标
     * @param businessType 业务线类型
     * @return 指标sql片段
     */
    private static String getStringMetricSumSql(String metric, int businessType) {
        StringBuilder sb = new StringBuilder();

        String resultFieldName = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metric)).getResultFieldName();
        sb.append("sum(").append(resultFieldName).append(")").append(" as ").append(resultFieldName).append(",");

        return sb.toString();
    }

    /**
     * 组装计算类指标sql片段
     *
     * @param metric       计算类指标
     * @param businessType 业务线类型
     * @return 计算类指标sql片段
     */
    private static String getCalculateMetricSumSql(String metric, int businessType) {
        StringBuilder sb = new StringBuilder();

        CalculateMetricEnum calculateMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, metric));

        sb.append(calculateMetricEnum.getMetricSql()).append(" as ").append(calculateMetricEnum.getResultFieldName()).append(",");

        return sb.toString();
    }

    /**
     * 组装组合指标sql片段
     *
     * @param metric       组合指标
     * @param businessType 业务线类型
     * @return 组合指标sql片段
     */
    private static String getCombineMetricSumSql(String metric, int businessType) {
        StringBuilder sb = new StringBuilder();

        CombineMetricEnum combineMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, metric));

        sb.append(combineMetricEnum.getMetricSql()).append(" as ").append(combineMetricEnum.getResultFieldName()).append(",");

        return sb.toString();
    }

    /**
     * 获取group by时的维度sql段
     *
     * @param dimensionList 维度列表
     * @param businessType  业务线类型
     * @return sql段
     */
    private static String getGroupBySumSql(String dataShowType, List<String> dimensionList, int businessType) {
        StringBuilder sb = new StringBuilder();

        //数据展示格式
        if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
            sb.append(HologresShowTypeEnum.valueOf(dataShowType).getFieldName()).append(", ");
        }

        sb.append(getDimensionSumSql(dimensionList, businessType));

        String sql = sb.toString();

        if (StringUtils.isNotBlank(sql)) {
            //去除最后的 ", "
            sql = sql.substring(0, sql.length() - 2);
        }

        return sql;
    }

    /**
     * 获取having 过滤条件
     *
     * @param metricConditionList 指标过滤条件
     * @param businessType        业务线类型
     * @return having sql
     */
    private static String getHavingSumSql(List<KeyValueDto> metricConditionList, int businessType) {
        if (CollectionUtils.isEmpty(metricConditionList)) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        for (KeyValueDto dto : metricConditionList) {
            String symbol = OperatorsEnum.valueOf(dto.getSymbol()).getSymbol();
            String value = dto.getValue().toString();

            if (Objects.requireNonNull(BusinessEnumUtil.getMetricEnumList(businessType)).contains(dto.getKey().toString())) {
                MetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, dto.getKey().toString()));
                sb.append("sum(").append(metricEnum.getResultFieldName()).append(")").append(symbol).append(value).append(" and ");
            } else if (Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnumList(businessType)).contains(dto.getKey().toString())) {
                CalculateMetricEnum calculateMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, dto.getKey().toString()));
                sb.append(calculateMetricEnum.getMetricSql()).append(symbol).append(value).append(" and ");
            } else if (Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnumList(businessType)).contains(dto.getKey().toString())) {
                CombineMetricEnum combineMetricEnum = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, dto.getKey().toString()));
                sb.append(combineMetricEnum.getMetricSql()).append(symbol).append(value).append(" and ");
            }
        }

        //删除最后一个多余的 "and"
        sb.delete(sb.length() - " and ".length(), sb.length());

        return sb.toString();
    }

    /**
     * 获取order by sql段
     *
     * @param orderByMetric 排序指标
     * @param businessType  业务线类型
     * @return sql段
     */
    private static String getOrderBySql(String dataShowType, SortDto orderByMetric, int businessType) {
        StringBuilder sb = new StringBuilder();

        if (orderByMetric == null) {
            if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
                sb.append(" order by ").append(HologresShowTypeEnum.valueOf(dataShowType).getFieldName()).append(" ").append(SortEnum.ASC.getName());
            }

            return sb.toString();
        }

        String orderField = getOrderField(orderByMetric.getKey(), businessType);

        if (StringUtils.isBlank(orderField)) {
            return sb.toString();
        }

        //组装order by语句
        sb.append(" order by ")
                .append(orderField)
                .append(" ")
                .append(SortEnum.valueOf(orderByMetric.getValue()).getName());
        return sb.toString();
    }

    /**
     * 获取排序字段
     *
     * @param key          字段名称
     * @param businessType 业务线类型
     * @return 排序字段
     */
    private static String getOrderField(String key, int businessType) {
        //基础指标
        List<String> metricEnumList = Objects.requireNonNull(BusinessEnumUtil.getMetricEnumList(businessType));
        if (metricEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, key)).getResultFieldName();
        }

        //计算类指标
        List<String> calculateMetricEnumList = Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnumList(businessType));
        if (calculateMetricEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getCalculateMetricEnum(businessType, key)).getResultFieldName();
        }

        //组合指标
        List<String> combineMetricEnumList = Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnumList(businessType));
        if (combineMetricEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getCombineMetricEnum(businessType, key)).getResultFieldName();
        }

        //维度字段
        List<String> dimensionEnumList = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnumList(businessType));
        if (dimensionEnumList.contains(key)) {
            return Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, key)).getResultFieldName();
        }

        return null;
    }

    /**
     * 组装所有指标对应的查询sql
     *
     * @param form         查询条件
     * @param metricList   指标列表
     * @param businessType 业务线类型
     * @return 查询sql
     * @throws Exception Exception
     */
    private static String getQueryUnionAllSql(HoloQueryForm form, List<Object> metricList, int businessType) throws Exception {
        StringBuilder sql = new StringBuilder();

        for (Object metric : metricList) {
            String querySql = SqlService.getQuerySql(form, metricList, metric, businessType);
            sql.append(querySql);
            sql.append(" union all ");
        }

        //删除最后一个多余的 "union all"
        sql.delete(sql.length() - " union all ".length(), sql.length());

        return sql.toString();
    }


}
