package cn.com.duiba.bigdata.common.biz.service;

import cn.com.duiba.bigdata.common.biz.dto.KeyValueDto;
import cn.com.duiba.bigdata.common.biz.enums.HologresShowTypeEnum;
import cn.com.duiba.bigdata.common.biz.enums.OperatorsEnum;
import cn.com.duiba.bigdata.common.biz.enums.TuiaDimensionEnum;
import cn.com.duiba.bigdata.common.biz.enums.tables.HologresTableEnum;
import cn.com.duiba.bigdata.common.biz.form.HoloQueryForm;
import cn.com.duiba.bigdata.common.biz.interfaces.DimensionEnum;
import cn.com.duiba.bigdata.common.biz.interfaces.MetricEnum;
import cn.com.duiba.bigdata.common.biz.utils.BusinessEnumUtil;
import cn.com.duiba.bigdata.common.biz.utils.DateFormatUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Objects;

/**
 * @author xugf 2023-11-28
 * 通过查询条件，自动组装查询sql（单个sql）
 */
@Slf4j
public class SqlService {

    //实验计划id
    public static final String ABTEST_PLAN_ID = TuiaDimensionEnum.ABTEST_PLAN_ID.toString();
    //实验分组id
    public static final String ABTEST_GROUP_ID = TuiaDimensionEnum.ABTEST_GROUP_ID.toString();
    //实验计划和分组
    public static final String ABTEST = TuiaDimensionEnum.ABTEST.toString();

    /**
     * 组装查询sql语句
     *
     * @param form         查询条件
     * @param metricList   查询的指标列表
     * @param metric       指标
     * @param businessType 业务线类型
     * @return sql
     * @throws Exception 异常
     */
    public static String getQuerySql(HoloQueryForm form, List<Object> metricList, Object metric, int businessType) throws Exception {
        //构建查询sql
        StringBuilder sb = new StringBuilder("select ");

        //查询维度
        String dimensionSql = getDimensionSql(form.getDataShowType(), form.getDimensionList(), form.getConditionList(), businessType);
        if (StringUtils.isNotBlank(dimensionSql)) {
            sb.append(dimensionSql);
        }

        //查询指标
        String metricSql = getMetricSql(metricList, metric, businessType);
        sb.append(metricSql);

        //holo 表
        String tableName = getHologresTable(metric, businessType);
        sb.append(" from ").append(tableName);

        //查询条件
        //时间（必填）
        long startTimeLong = DateFormatUtil.parse("yyyy-MM-dd HH:mm:ss", form.getStartTime()).getTime() / 1000;
        long endTimeLong = DateFormatUtil.parse("yyyy-MM-dd HH:mm:ss", form.getEndTime()).getTime() / 1000;
        long startDay = Long.parseLong(form.getStartTime().substring(0, 10).replaceAll("-", ""));
        long endDay = Long.parseLong(form.getEndTime().substring(0, 10).replaceAll("-", ""));

        //分区条件
        sb.append(" where date_partition >= ")
                .append(startDay)
                .append(" and date_partition <= ")
                .append(endDay);

        //明细表精确时间过滤条件
        if (HologresTableEnum.INSTANCE.exist(tableName, "UNIX_TIME")) {
            sb.append(" and unix_time >= ")
                    .append(startTimeLong)
                    .append(" and unix_time < ")
                    .append(endTimeLong);
        }

        //指标对应的条件
        String metricCondition = getMetricCondition(metric, businessType);
        if (StringUtils.isNotBlank(metricCondition)) {
            sb.append(" and ").append(metricCondition);
        }

        //页面传过来的条件
        String conditionSQL = getConditionSQL(form.getConditionList(), tableName, businessType);
        if (StringUtils.isNotBlank(conditionSQL)) {
            sb.append(conditionSQL);
        }

        //group by
        String groupBySql = getGroupBySql(form.getDataShowType(), form.getDimensionList(), businessType);
        if (StringUtils.isNotBlank(groupBySql)) {
            sb.append(" group by ").append(groupBySql);
        }

        return sb.toString();
    }

    /**
     * 维度列表对应的sql查询片段
     *
     * @param dimensionList 查询的维度列表
     * @param conditionList 用户输入的查询条件
     * @param businessType  业务线类型
     * @return sql查询片段
     */
    private static String getDimensionSql(String dataShowType, List<String> dimensionList, List<KeyValueDto> conditionList, int businessType) {
        StringBuilder sb = new StringBuilder();

        //数据展示格式
        if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
            HologresShowTypeEnum showTypeEnum = HologresShowTypeEnum.valueOf(dataShowType);
            sb.append(showTypeEnum.getSqlSegment()).append(" as ").append(showTypeEnum.getFieldName()).append(", ");
        }

        if (CollectionUtils.isEmpty(dimensionList)) {
            return sb.toString();
        }

        //实验平台字段
        if (dimensionList.contains(ABTEST_PLAN_ID)
                || dimensionList.contains(ABTEST_GROUP_ID)) {
            //查询维度中存在实验计划id和实验分组id，则在查询条件中一定有实验计划id和实验分组id，这个由前端进行联动管控
            //获取实验计划id和实验分组id
            String abtestValue = getABTestValue(conditionList);
            String[] array = StringUtils.split(abtestValue, "-");
            //实验计划id
            String abtestPlanId = array[0];
            //实验分组id
            String abtestGroupId = array[1];

            for (String dim : dimensionList) {
                DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dim));
                if (dim.equals(ABTEST_PLAN_ID)) {
                    sb.append(abtestPlanId).append(" as ").append(dimensionEnum.getResultFieldName()).append(", ");
                } else if (dim.equals(ABTEST_GROUP_ID)) {
                    sb.append(abtestGroupId).append(" as ").append(dimensionEnum.getResultFieldName()).append(", ");
                } else {
                    sb.append(dimensionEnum.getTableFieldName())
                            .append(" as ")
                            .append(dimensionEnum.getResultFieldName())
                            .append(", ");
                }
            }
        } else {
            for (String dim : dimensionList) {
                DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dim));
                sb.append(dimensionEnum.getTableFieldName())
                        .append(" as ")
                        .append(dimensionEnum.getResultFieldName())
                        .append(", ");
            }
        }

        return sb.toString();
    }

    /**
     * 获取实验计划id和实验分组id
     *
     * @param conditionList 用户输入的查询条件
     * @return 实验计划id和实验分组id
     */
    private static String getABTestValue(List<KeyValueDto> conditionList) {
        String value = "";
        for (KeyValueDto dto : conditionList) {
            if (dto.getKey().toString().equals(ABTEST)) {
                value = dto.getValue().toString();
            }
        }

        return value;
    }

    /**
     * 组装指标sql片段
     *
     * @param metricList   查询的指标列表
     * @param targetMetric 指标
     * @param businessType 业务线类型
     * @return 指标sql片段
     */
    private static String getMetricSql(List<Object> metricList, Object targetMetric, int businessType) {
        StringBuilder sb = new StringBuilder();
        for (Object metric : metricList) {
            sb.append(getMetricSql(metric, targetMetric, businessType));
        }

        return sb.deleteCharAt(sb.length() - 1).toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metric       指标
     * @param targetMetric 目标指标
     * @param businessType 业务线类型
     * @return 指标sql片段
     */
    private static String getMetricSql(Object metric, Object targetMetric, int businessType) {
        StringBuilder sb = new StringBuilder();

        if (metric instanceof String) {
            MetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metric.toString()));
            sb.append(getMetricSql(metric, targetMetric, metricEnum));
        } else if (metric instanceof List) {
            List<String> metricGroup = (List<String>) metric;
            for (String subMetric : metricGroup) {
                MetricEnum metricEnum = Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, subMetric));
                sb.append(getMetricSql(metric, targetMetric, metricEnum));
            }
        }

        return sb.toString();
    }

    /**
     * 组装指标sql片段
     *
     * @param metric       指标
     * @param targetMetric 目标指标
     * @param metricEnum   指标枚举类
     * @return 指标sql片段
     */
    private static String getMetricSql(Object metric, Object targetMetric, MetricEnum metricEnum) {
        StringBuilder sb = new StringBuilder();
        //指标查询语句
        String metricSql;

        //直接判断两个对象的内存地址是否相同
        if (metric != targetMetric) {
            metricSql = "0";
        } else {
            metricSql = metricEnum.getMetricSql();
        }

        sb.append(metricSql)
                .append(" as ")
                .append(metricEnum.getResultFieldName())
                .append(",");

        return sb.toString();
    }

    /**
     * 获取指标对应的数据库表
     *
     * @param metric       查询指标
     * @param businessType 业务线类型
     * @return 数据库表
     */
    private static String getHologresTable(Object metric, int businessType) {
        //指标名称
        String metricName = "";

        if (metric instanceof String) {
            metricName = metric.toString();
        } else if (metric instanceof List) {
            List<String> metricGroup = (List<String>) metric;
            metricName = metricGroup.get(0);
        }

        return Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metricName)).getTableName();
    }

    /**
     * 获取指标对应的查询条件
     *
     * @param metric       查询指标
     * @param businessType 业务线类型
     * @return 查询条件
     */
    private static String getMetricCondition(Object metric, int businessType) {
        if (metric instanceof String) {
            return Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metric.toString())).getCondition();
        } else if (metric instanceof List) {
            List<String> metricGroup = (List<String>) metric;
            return Objects.requireNonNull(BusinessEnumUtil.getMetricEnum(businessType, metricGroup.get(0))).getCondition();
        }

        return null;
    }

    /**
     * 组装查询条件
     *
     * @param conditionList 查询条件
     * @param tableName     指标对应的hologres表
     * @param businessType  业务线类型
     */
    private static String getConditionSQL(List<KeyValueDto> conditionList, String tableName, int businessType) {
        if (CollectionUtils.isEmpty(conditionList)) {
            return null;
        }

        StringBuilder sb = new StringBuilder();
        for (KeyValueDto pair : conditionList) {
            getConditionSQLValue(sb, pair, tableName, businessType);
        }

        return sb.toString();
    }

    /**
     * 组装单个查询条件
     *
     * @param sb           sql
     * @param pair         单个条件
     * @param tableName    指标对应的hologres表
     * @param businessType 业务线类型
     */
    private static void getConditionSQLValue(StringBuilder sb, KeyValueDto pair, String tableName, int businessType) {
        if (pair == null || pair.getKey() == null) {
            return;
        }

        String key = pair.getKey().toString();

        DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, key));
        if (!HologresTableEnum.INSTANCE.exist(tableName, key)) {
            //说明这个条件不适合这个指标，直接过滤掉
            return;
        }

        String tableFieldName = dimensionEnum.getTableFieldName();
        Object value = pair.getValue();
        String symbol = OperatorsEnum.valueOf(pair.getSymbol()).getSymbol();

        if (value == null || StringUtils.isBlank(value.toString())) {
            if (pair.getSymbol().equals(OperatorsEnum.IS_NULL.toString())
                    || pair.getSymbol().equals(OperatorsEnum.IS_NOT_NULL.toString())) {
                sb.append(" and ").append(tableFieldName).append(symbol);
            }
        } else {
            String pairValue = getPairValue(value, dimensionEnum.getFieldType());
            if (pair.getSymbol().equals(OperatorsEnum.IN.toString())
                    || pair.getSymbol().equals(OperatorsEnum.NOT_IN.toString())
                    || pair.getSymbol().equals(OperatorsEnum.ARRAY.toString())) {
                //赋值替换
                symbol = symbol.replace("${value}", pairValue);
                sb.append(" and ").append(tableFieldName).append(symbol);
            }
        }

    }

    /**
     * 获取条件对应的数值
     *
     * @param pairValue 条件
     * @return 条件对应的数值
     */
    private static String getPairValue(Object pairValue, String fieldType) {
        if (pairValue instanceof List) {
            List<Object> list = (List<Object>) pairValue;

            StringBuilder sb = new StringBuilder();
            for (Object value : list) {
                if ("String".equalsIgnoreCase(fieldType)
                        || "array-string".equalsIgnoreCase(fieldType)) {
                    sb.append("'").append(value.toString()).append("'").append(",");
                } else {
                    sb.append(value.toString()).append(",");
                }
            }
            sb.deleteCharAt(sb.length() - 1);
            return sb.toString();
        } else {
            if ("String".equalsIgnoreCase(fieldType)
                    || "array-string".equalsIgnoreCase(fieldType)) {
                return "'" + pairValue.toString() + "'";
            } else {
                return pairValue.toString();
            }
        }
    }

    /**
     * 获取group by时的维度sql段
     *
     * @param dataShowType  数据展示格式
     * @param dimensionList 维度列表
     * @param businessType  业务线类型
     * @return sql段
     */
    private static String getGroupBySql(String dataShowType, List<String> dimensionList, int businessType) {
        StringBuilder sb = new StringBuilder();

        //数据展示格式
        if (!HologresShowTypeEnum.ALL.toString().equals(dataShowType)) {
            sb.append(HologresShowTypeEnum.valueOf(dataShowType).getSqlSegment()).append(",");
        }

        if (CollectionUtils.isNotEmpty(dimensionList)) {
            for (String dim : dimensionList) {
                //排除掉实验计划id和实验分组id字段，数据库中并不存在这两个字段
                if (dim.equals(ABTEST_PLAN_ID) || dim.equals(ABTEST_GROUP_ID)) {
                    continue;
                }

                DimensionEnum dimensionEnum = Objects.requireNonNull(BusinessEnumUtil.getDimensionEnum(businessType, dim));
                sb.append(dimensionEnum.getTableFieldName()).append(",");
            }
        }

        if (StringUtils.isNotBlank(sb.toString())) {
            //去掉最后的 ","
            sb.deleteCharAt(sb.length() - 1);
        }

        return sb.toString();
    }

}
