package cn.com.duiba.biz.tool.duiba.client;

import cn.com.duiba.activity.center.api.remoteservice.activity.RemoteAppDirectService;
import cn.com.duiba.biz.tool.duiba.dto.ConsumerCookieDto;
import cn.com.duiba.biz.tool.duiba.qpslimit.QpsLimitProxy;
import cn.com.duiba.consumer.center.api.dto.ConsumerDto;
import cn.com.duiba.consumer.center.api.remoteservice.RemoteConsumerService;
import cn.com.duiba.developer.center.api.domain.dto.AppSimpleDto;
import cn.com.duiba.developer.center.api.remoteservice.RemoteAppService;
import cn.com.duiba.idmaker.service.api.remoteservice.kms.RemoteKmsService;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.BlowfishUtils;
import cn.com.duiba.wolf.utils.UUIDUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Optional;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import javax.servlet.ServletException;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

/**
 * 使用方法，在spring配置文件中声明即可:<bean id="requestLocal" class="cn.com.duiba.biz.tool.duiba.client.RequestLocal"/>
 * 实例化时会自动寻找RemoteAppService、RemoteConsumerService并注入，找不到则报错要求声明这两个bean
 *
 * Created by wenqi.huang on 2017/5/12.
 */
public class RequestLocal implements ApplicationContextAware, InitializingBean{

    private static final Logger logger = LoggerFactory.getLogger(RequestLocal.class);

    private ApplicationContext applicationContext;

    private static RemoteConsumerService remoteConsumerService;
    private static RemoteAppService remoteAppService;
    private static RemoteAppDirectService remoteAppDirectService;
    private static DuibaConsumerCookieClient duibaConsumerCookieClient;

    //这个是用于兼容老的固定密钥加解密逻辑，以及商业流量
    private static String oldConsumerEncryptKey;
    //cookie的domain
    private static String cookieDomain;

    //限流器
    private static QpsLimitProxy qpsLimitProxy;

    private static LoadingCache<String, Optional<Long>> appKey2IdCache;
    private static LoadingCache<Long, Boolean> appId2DirectCache;

    private static ThreadLocal<HttpRequestDto> local = new ThreadLocal<>();

    static{
        appKey2IdCache = CacheBuilder.newBuilder().concurrencyLevel(32).softValues().maximumSize(100000)
                .refreshAfterWrite(8, TimeUnit.MINUTES)
                .expireAfterWrite(10, TimeUnit.MINUTES)
                .build(new CacheLoader<String, Optional<Long>>() {
                    @Override
                    public Optional<Long> load(String key) throws Exception {
                        AppSimpleDto app = remoteAppService.getAppByAppKey(key).getResult();
                        if(app != null){
                            return Optional.of(app.getId());
                        }
                        return Optional.absent();
                    }
                });
        appId2DirectCache = CacheBuilder.newBuilder().concurrencyLevel(32).softValues().maximumSize(100000)
                .refreshAfterWrite(25, TimeUnit.MINUTES)//25分钟后异步刷新
                .expireAfterWrite(30, TimeUnit.MINUTES)//30分钟后失效
                .build(new CacheLoader<Long, Boolean>() {
                    @Override
                    public Boolean load(Long key) throws Exception {
                        Long id = remoteAppDirectService.findByAppId(key).getResult();
                        if(id != null){
                            return true;
                        }
                        return false;
                    }
                });
    }

    private static class HttpRequestDto {

        private ConsumerDto consumerDO;
        private AppSimpleDto consumerAppDO;
        private ConsumerCookieDto consumerCookieDto;//cookie中直接获取的信息
        private String tokenId;
        private HttpServletRequest request;
        private HttpServletResponse response;

        /**
         * 尝试从request中恢复用户信息，并限流
         * @param request
         * @param response
         * @return true表示此次请求被限流，用户会看到繁忙页面；false表示此次请求被放行。
         * @throws IOException
         * @throws ServletException
         */
        public boolean initAndLimitQps(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
            this.request = request;
            this.response = response;

            ConsumerCookieDto consumerFromWdata4 = null;
            ConsumerCookieDto consumerFromWdata3 = null;
            //先尝试从wdata4中获取用户信息
            try {
                consumerFromWdata4 = duibaConsumerCookieClient.getConsumerCookieDto(request);
            }catch(Exception e){
                logger.error("get user from wdata4 error,will try to use wdata3.", e);
            }
            //如果取不到用户信息，则尝试从wdata3中获取用户信息
            if(consumerFromWdata4 == null){
                consumerFromWdata3 = parseOldWdata3();
            }

            this.tokenId = RequestTool.getCookie(request, DuibaConsumerCookieClient.TOKEN_ID_COOKIE);
            //TODO to be deleted start
            if(this.tokenId == null || this.tokenId.isEmpty()){
                this.tokenId = RequestTool.getCookie(request, "wdata3");
            }
            //TODO to be deleted end

            String appKey=request.getParameter("appKey");
            String openBs=request.getParameter("openBs");
            //如果是商业活动，则可能需要重新植入用户cookie
            boolean preferUseWdata4 = false;
            boolean qpsLimitInvoked = false;
            if(null != appKey && "openbs".equals(openBs)){
                Optional<Long> appIdOptional = appKey2IdCache.getUnchecked(appKey);
                Long appId = appIdOptional.isPresent() ? appIdOptional.get() : null;
                //判断是否商业活动
                if(null!=appId && isAppDirectByAppId(appId)){
                    this.consumerCookieDto = consumerFromWdata3;
                    removeWdata4Cookie();//如果是商业流量，需要确保移除wdata4 cookie，只保留wdata3，以防下次访问其他页面使用了wdata4 cookie
                    if((consumerCookieDto != null && !appId.equals(consumerCookieDto.getAppId()))
                            || consumerCookieDto == null){
                        //对app限流
                        if(qpsLimitProxy.doLimit(request, response, null, appId)){
                            return true;//被限流则直接返回,避免创建用户
                        }
                        qpsLimitInvoked = true;
                        makeCommercialUser(appId);
                    }
                }else{
                    preferUseWdata4 = true;
                }
            }else{
                preferUseWdata4 = true;
            }
            if(preferUseWdata4){
                this.consumerCookieDto = consumerFromWdata4 == null ? consumerFromWdata3 : consumerFromWdata4;
            }
            if(!qpsLimitInvoked && consumerCookieDto != null){
                //限流
                String limitConsumerId = RequestLocal.isLoginConsumer()?String.valueOf(consumerCookieDto.getCid()):null;
                return qpsLimitProxy.doLimit(request, response, limitConsumerId, consumerCookieDto.getAppId());
            }else if(!qpsLimitInvoked && consumerCookieDto == null){
                //未登录状态下，尝试从appKey中识别出appId（通常这是个登录请求），并进行限流，这样做是为了限制某个app的登录请求，防止有开发商用大量用户分布式恶意刷券（这样做有个小缺陷，在对这个app开启限流的情况下，其他用户可以狂刷有这个appKey的链接，会影响真实的流量）
                Optional<Long> appIdOptional = appKey2IdCache.getUnchecked(appKey);
                Long appId = appIdOptional.isPresent() ? appIdOptional.get() : null;
                return qpsLimitProxy.doLimit(request, response, null, appId);
            }
            //TODO 对于未登录用户，考虑是否按照ip限流？（对于有appId却没有cid的情况也可以这样做）
            return false;
        }

        //移除wdata4 cookie
        private void removeWdata4Cookie(){
            for(Cookie cookie : this.request.getCookies()){
                if(cookie.getName().equals(DuibaConsumerCookieClient.CONSUMER_WDATA4_COOKIE)
                        || cookie.getName().equals(DuibaConsumerCookieClient.LOGIN_TIME_COOKIE)){
                    cookie.setValue("");
                    cookie.setMaxAge(0);
                    cookie.setDomain(cookieDomain);
                    cookie.setPath("/");
                    cookie.setHttpOnly(true);
                    response.addCookie(cookie);
                }
            }
        }

        //创建商业流量用户并植入cookie
        private void makeCommercialUser(Long appId){
            String uid = "gen_" + UUIDUtils.createUUID();
            ConsumerDto c = new ConsumerDto(true);
            c.setAppId(appId);
            c.setPartnerUserId(uid);

            ConsumerDto temp = remoteConsumerService.insert(c);
            c.setId(temp.getId());

            RequestLocal.injectConsumerInfoIntoCookie(c, true);
        }

        //判断是否商业流量app
        private Boolean isAppDirectByAppId(Long appId) {
            try {
                return appId2DirectCache.get(appId);
            } catch (ExecutionException e) {
                throw new RuntimeException(e);
            }
        }

        //解析旧的用户cookie数据，或者商业流量数据
        private ConsumerCookieDto parseOldWdata3(){
            String wdata3 = RequestTool.getCookie(request, "wdata3");
            if(wdata3 == null || wdata3.isEmpty()){
                return null;
            }
            String content = BlowfishUtils.decryptBlowfish(wdata3, oldConsumerEncryptKey);
            ConsumerCookieDto c = JSONObject.parseObject(content, ConsumerCookieDto.class);
            //如果是商业流量，则cookie永久有效,仍然使用wdata3
            if(isAppDirectByAppId(c.getAppId())){
                return c;
            }else {//TODO 这个else分支等到下次发布需要删除，非商业流量只应该使用wdata4 cookie,目前存在只是为了兼容
                //非商业流量，cookie24小时内有效
                long now = System.currentTimeMillis();
                if (c.getTime() > now - 86400000) {
                    return c;
                }
            }
            return null;
        }

        //延迟初始化
        public ConsumerDto getConsumerDO(){
            if(consumerDO == null){
                if(consumerCookieDto == null){//如果cookie中没有有效的用户信息，则直接返回
                    return null;
                }else{
                    //返回经过代理的对象，获取appId/id/partnerUserId字段时不会读取数据库，获取其他字段才会从数据库加载
                    ConsumerDto temp = new ConsumerDto();
                    temp.setAppId(consumerCookieDto.getAppId());
                    temp.setId(consumerCookieDto.getCid());
                    temp.setPartnerUserId(consumerCookieDto.getPartnerUserId());

                    ProxyFactory proxyFactory = new ProxyFactory();
                    proxyFactory.setTarget(temp);
                    proxyFactory.addAdvice(new ConsumerMethodInterceptor(this));
                    consumerDO = (ConsumerDto)proxyFactory.getProxy();
                }
            }

            return consumerDO;
        }

        //延迟初始化
        public AppSimpleDto getConsumerAppDO() {
            if (consumerAppDO == null) {
                if(consumerCookieDto == null){//如果cookie中没有有效的用户信息，则直接返回
                    return null;
                }else{
                    //返回经过代理的对象，获取id字段时不会读取数据库，获取其他字段才会从数据库加载
                    AppSimpleDto temp = new AppSimpleDto();
                    temp.setId(consumerCookieDto.getAppId());

                    ProxyFactory proxyFactory = new ProxyFactory();
                    proxyFactory.setTarget(temp);
                    proxyFactory.addAdvice(new AppMethodInterceptor(this));
                    consumerAppDO = (AppSimpleDto)proxyFactory.getProxy();
                }
            }
            return consumerAppDO;
        }

        public HttpServletResponse getResponse() {
            if(response == null){
                throw new IllegalStateException("response must not be null, please invoke RequestLocal.setThreadLocallyAndLimitQps first");
            }
            return response;
        }
        public HttpServletRequest getRequest() {
            if(request == null){
                throw new IllegalStateException("request must not be null, please invoke RequestLocal.setThreadLocallyAndLimitQps first");
            }
            return request;
        }

        public String getTokenId() {
            return tokenId;
        }
    }

    private static class ConsumerMethodInterceptor implements MethodInterceptor{

        private boolean isLazyInitted = false;
        private HttpRequestDto httpRequestDto;

        public ConsumerMethodInterceptor(HttpRequestDto httpRequestDto){
            this.httpRequestDto = httpRequestDto;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(isLazyInitted){
                return invocation.proceed();
            }

            Method m = invocation.getMethod();
            if(m.getName().equals("getId") || m.getName().equals("getAppId") || m.getName().equals("getPartnerUserId")) {//m.getName().equals("isLazyInitted")
                //如果调用已经有了预置属性的方法，直接调用
                return invocation.proceed();
            }
            //调用其他get方法则从远程获取数据，并设置originalConsumer的属性。
            ConsumerDto originalConsumer = (ConsumerDto)invocation.getThis();
            ConsumerDto consumerInDb = remoteConsumerService.find(originalConsumer.getId());

            //可能调用方仍然持有proxy的实例，所以需要设置原始属性
            org.springframework.beans.BeanUtils.copyProperties(consumerInDb, originalConsumer);
            isLazyInitted = true;
            //同时改变原始引用为originalConsumer，加速下次的访问（下次访问不用再经过代理了）
            httpRequestDto.consumerDO = originalConsumer;

            return invocation.proceed();
        }
    }

    private static class AppMethodInterceptor implements MethodInterceptor{

        private boolean isLazyInitted = false;
        private HttpRequestDto httpRequestDto;

        public AppMethodInterceptor(HttpRequestDto httpRequestDto){
            this.httpRequestDto = httpRequestDto;
        }

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if(isLazyInitted){
                return invocation.proceed();
            }

            Method m = invocation.getMethod();
            if(m.getName().equals("getId")) {//m.getName().equals("isLazyInitted")
                //如果调用已经有了预置属性的方法，直接调用
                return invocation.proceed();
            }
            //调用其他get方法则从远程获取数据，并设置originalConsumer的属性。
            AppSimpleDto originalApp = (AppSimpleDto)invocation.getThis();
            AppSimpleDto appInDb = remoteAppService.getSimpleApp(originalApp.getId()).getResult();

            //可能调用方仍然持有proxy的实例，所以需要设置原始属性
            org.springframework.beans.BeanUtils.copyProperties(appInDb, originalApp);
            isLazyInitted = true;
            //同时改变原始引用为originalConsumer，加速下次的访问（下次访问不用再经过代理了）
            httpRequestDto.consumerAppDO = originalApp;

            return invocation.proceed();
        }
    }

    private static HttpRequestDto get() {
        HttpRequestDto rl = local.get();
        if (rl == null) {
            rl = new HttpRequestDto();
            local.set(rl);
        }
        return rl;
    }

    /**
     * 获得当前登录的用户id
     * @return
     */
    public static Long getCid(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getCid();
    }

    /**
     * 获得当前登录的用户的appId
     * @return
     */
    public static Long getAppId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getAppId();
    }

    /**
     * 获得当前登录的用户的partnerUserId
     * @return
     */
    public static String getPartnerUserId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.consumerCookieDto == null ? null : requestLocal.consumerCookieDto.getPartnerUserId();
    }

    /**
     * 获取当前登录的用户信息,获取appId/id/partnerUserId字段时不会读取数据库，获取其他字段才会从数据库加载
     * @return
     */
    public static ConsumerDto getConsumerDO() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerDO();
    }

    /**
     * 获取当前登录的用户所在的app信息,获取id字段时不会读取数据库，获取其他字段才会从数据库加载
     * @return
     */
    public static AppSimpleDto getConsumerAppDO() {
        HttpRequestDto requestLocal = get();

        return requestLocal.getConsumerAppDO();
    }

    /**
     * 获取tokenId， tokenId的生命周期和wdata4相同，这个tokenId是唯一可以在js里直接获得的用户相关的cookie，（其他用户cookie全都设为了httpOnly，防止被XSS攻击获得）；作用主要有：用于传递给同盾做会话id。
     * @return
     */
    public static String getTokenId(){
        HttpRequestDto requestLocal = get();

        return requestLocal.getTokenId();
    }

    /**
     * 当前用户是否为登录状态(如果用户的partnerUserId为not_login也视为未登录状态)
     * @return
     */
    public static boolean isLoginConsumer(){
        String partnerUserId = getPartnerUserId();
        return partnerUserId != null && !ConsumerDto.NOTLOGINUSERID.equals(partnerUserId);
    }

    /**
     * 当前用户是否为分享用户
     * @return
     */
    public static boolean isShareConsumer(){
        return ConsumerDto.SHAREUSERID.equals(getPartnerUserId());
    }

    /**
     * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @return 设置到cookie中的用户信息
     */
    public static void injectConsumerInfoIntoCookie(ConsumerDto consumer){
        injectConsumerInfoIntoCookie(consumer, false);
    }

    /**
     * 登录请求，用于把用户信息放入cookie(包括wdata4、w_ts、_ac、tokenId四个cookie)，调用此方法会顺便把用户信息放到ThreadLocal中，以便后续请求能获取到正确的用户信息
     * @param consumer
     * @param isFromCommercial 是否来自商业流量, 如果是商业流量，则用户信息会放入wdata3中，永久有效；否则用户信息放入wdata4中，24小时有效
     * @return 设置到cookie中的用户信息
     */
    private static void injectConsumerInfoIntoCookie(ConsumerDto consumer, boolean isFromCommercial){
        HttpRequestDto requestLocal = get();

        ConsumerCookieDto consumerCookieDto = duibaConsumerCookieClient.injectConsumerInfoIntoCookie(consumer, requestLocal.getRequest(), requestLocal.getResponse(), cookieDomain, oldConsumerEncryptKey, isFromCommercial);

        //重置当前登录用户的属性
        requestLocal.consumerCookieDto = consumerCookieDto;
        requestLocal.consumerDO = null;
        requestLocal.consumerAppDO = null;
        requestLocal.tokenId = (String)requestLocal.getRequest().getAttribute("c_tokenId");
    }

    /**
     * 设置request和response ，必须在获取用户信息之前调用（一般是在LoginFilter开始时）,内部会自动解析出用户信息，如果是商业流量，还会自动设置永久的用户cookie
     * @param request
     * @param response
     * @return true表示此次请求被限流，用户会看到繁忙页面,调用方应当判断返回值，为true时应该中断处理流程；false表示此次请求被放行。
     */
    public static boolean setThreadLocallyAndLimitQps(HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
        if(request == null || response == null){
            throw new IllegalArgumentException("request or response must not be null");
        }

        HttpRequestDto dto = get();
        return dto.initAndLimitQps(request, response);
    }

    /**
     * 清空线程本地变量,必须在处理完用户请求之后调用（一般是在LoginFilter调用链结束时）
     */
    public static void clearThreadLocally() {
        local.remove();
    }

    /**
     * 销毁工作，在LoginFilter.destroy中调用，此方法调用后RequestLocal不再提供服务。
     */
    public static void destroy() {
        qpsLimitProxy.destroy();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        RemoteAppService remoteAppService = applicationContext.getBean(RemoteAppService.class);
        if(remoteAppService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteAppService(in developer-center)");
        }
        RemoteConsumerService remoteConsumerService = applicationContext.getBean(RemoteConsumerService.class);
        if(remoteAppService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteConsumerService(in consumer-center)");
        }
        RemoteKmsService remoteKmsService = applicationContext.getBean(RemoteKmsService.class);
        if(remoteKmsService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteKmsService(in idmaker-service)");
        }
        RemoteAppDirectService remoteAppDirectService = applicationContext.getBean(RemoteAppDirectService.class);
        if(remoteAppDirectService == null){
            throw new IllegalStateException("there must exists a bean of class RemoteAppDirectService(in activity-center)");
        }
        oldConsumerEncryptKey = applicationContext.getEnvironment().getProperty("app.consumer.encrypt.key");
        if(oldConsumerEncryptKey == null || oldConsumerEncryptKey.isEmpty()){
            throw new IllegalStateException("property:[app.consumer.encrypt.key] does not exist");
        }
        //zookeeper地址
        String zookeeperAddress = applicationContext.getEnvironment().getProperty("app.zookeeper.address");
        if(zookeeperAddress == null || zookeeperAddress.isEmpty()){
            throw new IllegalStateException("property:[app.zookeeper.address] does not exist");
        }
        cookieDomain = applicationContext.getEnvironment().getProperty("app.cross.domain");
        if(cookieDomain == null || cookieDomain.isEmpty()){
            throw new IllegalStateException("property:[app.cross.domain] does not exist");
        }
        if(cookieDomain.startsWith(".") || cookieDomain.startsWith("-")){//不能以.或者-开头
            cookieDomain = cookieDomain.substring(1);
        }

        RequestLocal.qpsLimitProxy = new QpsLimitProxy(zookeeperAddress, 5);
        RequestLocal.remoteAppService = remoteAppService;
        RequestLocal.remoteConsumerService = remoteConsumerService;
        RequestLocal.remoteAppDirectService = remoteAppDirectService;
        RequestLocal.duibaConsumerCookieClient = new DuibaConsumerCookieClient(remoteKmsService);
    }
}
