package cn.com.duiba.biz.tool.duiba.qpslimit;

import com.alibaba.fastjson.JSONArray;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import org.I0Itec.zkclient.IZkDataListener;
import org.I0Itec.zkclient.ZkClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.*;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 针对某个App/requestUri的访问，如果超出设定的并发数，则进行限流,供兑吧和麦拉使用
 * 
 * http://cf.dui88.com/pages/viewpage.action?pageId=4361803
 *
 * 本类供给SpringMvc的Interceptor或者Servlet Filter使用
 *
 * @author huangwenqi
 *
 */
public class QpsLimitProxy {

	private static Logger log=LoggerFactory.getLogger(QpsLimitProxy.class);

	private static final int DEFAULT_USER_QPS_LIMIT = 15;//默认用户每秒最多15个请求

	private int userQpsLimit = DEFAULT_USER_QPS_LIMIT;//用户每秒最多N个请求

	private ZkClient zkClient;

	private LoadingCache<String,AtomicInteger> consumerId2QpsCache;

    /**
     * 表示缓存的qps数据保存几秒，默认是1, 实际判断qps限制时会乘以这个参数
     */
    private int qpsDataCacheSeconds;

	private ConcurrentMap<Long, ConcurrentMap<String,AppUriQpsLimit>> appId2uri2qpsLimitMap = new ConcurrentHashMap<Long, ConcurrentMap<String,AppUriQpsLimit>>();

	private LoadingCache<String,AtomicInteger> appIdUri2QpsCache;

	//用于控制一个app 在10秒内只允许打印一次qps warn日志
	private Cache<Long, Object> appId2WarnedFlagCache;

	private volatile boolean isDestroyed = false;

	public QpsLimitProxy(String zkServers){
	    this(zkServers, 1);
	}

    /**
     *
     * @param zkServers zookeeper服务器地址，多个用逗号隔开,如果需要设置在不同路径下，可以加入后缀，比如dev.config.duibar.com:2181/maila
     * @param qpsDataCacheSeconds 表示缓存的qps数据保存几秒，默认是1,取值范围1-10
     */
    public QpsLimitProxy(String zkServers, int qpsDataCacheSeconds){
    	this(zkServers, qpsDataCacheSeconds, 10);
    }

    /**
     *
     * @param zkServers zookeeper服务器地址，多个用逗号隔开
     * @param qpsDataCacheSeconds 表示缓存的qps数据保存几秒，默认是1,取值范围1-10
     * @param appWarnLogSilencePeriod 对于同一个appId的告警日志，N秒内最多输出一次，默认是10（即每个app10秒内最多只允许输出一次warn日志）
     */
    public QpsLimitProxy(String zkServers, int qpsDataCacheSeconds, int appWarnLogSilencePeriod){
        if(qpsDataCacheSeconds < 1 || qpsDataCacheSeconds > 10){
            throw new IllegalArgumentException("qpsDataCacheSeconds must between 1 and 10");
        }
        this.qpsDataCacheSeconds = qpsDataCacheSeconds;
        zkClient = new ZkClient(zkServers);

        readQpsLimitData("/QpsLimit");
        zkClient.subscribeDataChanges("/QpsLimit", new IZkDataListener() {
            @Override
            public void handleDataChange(String parentPath, Object parentData) throws Exception {
                readQpsLimitData(parentPath);
            }

            @Override
            public void handleDataDeleted(String dataPath) throws Exception {
            }
        });

        consumerId2QpsCache = CacheBuilder.newBuilder()
                .expireAfterWrite(qpsDataCacheSeconds,TimeUnit.SECONDS)
                .concurrencyLevel(4)
                .initialCapacity(10000)
                .maximumSize(1000000)//最多存100W个用户信息，大约20M内存
                .build(new CacheLoader<String, AtomicInteger>() {
                    @Override
                    public AtomicInteger load(String key) throws Exception {
                        return new AtomicInteger(0);
                    }
                });

        appIdUri2QpsCache = CacheBuilder.newBuilder()
                .expireAfterWrite(qpsDataCacheSeconds,TimeUnit.SECONDS)
                .concurrencyLevel(4)
                .initialCapacity(10000)
                .maximumSize(1000000)//最多存100W个用户信息，大约20M内存
                .build(new CacheLoader<String, AtomicInteger>() {
                    @Override
                    public AtomicInteger load(String key) throws Exception {
                        return new AtomicInteger(0);
                    }
                });

		appId2WarnedFlagCache = CacheBuilder.newBuilder()
                .expireAfterWrite(appWarnLogSilencePeriod,TimeUnit.SECONDS)
                .concurrencyLevel(4)
                .initialCapacity(100)
                .maximumSize(1000000)//最多存100W个app信息，大约20M内存
                .build();
    }

	private void readQpsLimitData(String parentPath){
		if(zkClient.exists(parentPath)) {
			List<String> list = zkClient.getChildren(parentPath);
			for (String path : list) {
				if (path == null) {//parentPath
					continue;
				}

				String data = zkClient.readData(parentPath + "/" + path, true);
				Long appId = Long.valueOf(path);
				if (data == null) {//node not exists
					appId2uri2qpsLimitMap.remove(appId);
				} else {
					List<AppUriQpsLimit> limitList = JSONArray.parseArray(data, AppUriQpsLimit.class);
					if (limitList == null || limitList.isEmpty()) {
						appId2uri2qpsLimitMap.remove(appId);
						continue;
					}
					ConcurrentMap<String, AppUriQpsLimit> uri2qpsLimitMap = appId2uri2qpsLimitMap.get(appId);
					if (uri2qpsLimitMap == null) {
						uri2qpsLimitMap = new ConcurrentHashMap<String, AppUriQpsLimit>();
					}

					Date now = new Date();
					for (AppUriQpsLimit limit : limitList) {
						if (limit.getFinishTime().after(now)) {//只添加有效的
							uri2qpsLimitMap.put(limit.getLimitUri(), limit);
						}
					}

					appId2uri2qpsLimitMap.put(appId, uri2qpsLimitMap);
				}
			}
		}
	}

	/**
	 * 在拦截器中调用该方法，进行限流
	 * @param request
	 * @param response
	 * @param consumerId 用户id或者设备id
	 * @param appId appId
	 * @return 是否限流，如果限流,本方法内部会负责把系统繁忙页面输出到response，使用者不应该继续Filter或Inteceptor链路处理。
	 * @throws IOException
	 * @throws ServletException
	 */
	public boolean doLimit(HttpServletRequest request, HttpServletResponse response,
						 String consumerId, Long appId) throws IOException, ServletException {
		if(isDestroyed){
			throw new IllegalStateException("QpsLimitProxy is destroyed, can not provide service any more");
		}
		if (consumerId != null && !isTestMode(request)) {//压测模式不限制用户级的qps，方便进行压测
			//针对用户限流
			boolean limited = limitByConsumer(consumerId, response);
			if(limited){
				return true;
			}
		}

		if(appId != null) {
			String requestUri = request.getRequestURI();
			//针对单个app限流
			boolean limited = limitByAppAndUri(appId, requestUri, response);
			if(limited){
				return true;
			}

			//针对全部app限流
			limited = limitByAppAndUri(0L, requestUri, response);
			if(limited){
				return true;
			}
		}

		return false;
	}

	/**
	 * 配合兑吧的 spring-boot-starter-test 线上压测项目而写的方法，判断到url参数包含_duiba_test=1或者cookie里包含_duiba_test=1则认为是压测模式，不应该对压测流量做用户级的qps限制。
	 * @param request
	 * @return
	 */
	private boolean isTestMode(HttpServletRequest request){
		boolean isTestMode = false;
		String testInParameter = request.getParameter("_duibaPerf");
		if(testInParameter != null && ("1".equals(testInParameter) || "true".equals(testInParameter))){
			isTestMode = true;
		}else{
			Cookie[] cookies = request.getCookies();
			if(cookies != null) {
				for (Cookie cookie : cookies) {
					if("_duibaPerf".equals(cookie.getName()) && ("1".equals(cookie.getValue()) || "true".equals(cookie.getValue()))){
						isTestMode = true;
					}
				}
			}
		}
		return isTestMode;
	}

	private boolean limitByAppAndUri(final Long appId, String requestUri, HttpServletResponse response) throws IOException {
		AppUriQpsLimit qpsLimit = getAppUriQpsLimit(appId, requestUri);
		if(qpsLimit == null){
			return false;
		}
		Integer qpsLimitInteger = qpsLimit.getQpsLimit();
		if(qpsLimitInteger <= 0){
			showBusyPage(response);//qps限制<=0表示直接禁止访问
			return true;
		}else{
			try {
				AtomicInteger qpsAtomic = appIdUri2QpsCache.get(appId + "-" + requestUri);
				int qps = qpsAtomic.get();
				if(qps >= qpsLimitInteger * qpsDataCacheSeconds){
					showBusyPage(response);

					//同一个app10秒内只允许输出一次warn日志
					final AtomicBoolean canLog = new AtomicBoolean(false);
					appId2WarnedFlagCache.get(appId, new Callable<Object>() {
						@Override
						public Object call() throws Exception {
							canLog.set(true);
							return appId;
						}
					});
					if(canLog.get()) {
						log.warn("app:{},uri:{}, qps:{},超出{}", appId, requestUri, qps, qpsLimitInteger);
					}
					return true;
				}

				qpsAtomic.incrementAndGet();
			} catch (ExecutionException e) {
				log.error("",e);
			}
		}
		return false;
	}

	/**
	 *
	 * @param consumerId
	 * @param response
	 * @return
	 * @throws IOException
	 */
	private boolean limitByConsumer(String consumerId, HttpServletResponse response) throws IOException {
		try {
			AtomicInteger qpsAtomic = consumerId2QpsCache.get(consumerId);
			int qps = qpsAtomic.get();
            int limit = qpsDataCacheSeconds * userQpsLimit;
			if(qps >= limit){
				showBusyPage(response);
				log.warn("user:{} qps:{},超出{}", consumerId,qps, userQpsLimit);
				return true;
			}

			qpsAtomic.incrementAndGet();
		} catch (ExecutionException e) {
			log.error("",e);
		}

		return false;
	}

	/**
	 * 获得appId对应的限制，appId为0表示对全部app生效。
	 * @param appId
	 * @param requestUri
	 * @return
	 */
	private AppUriQpsLimit getAppUriQpsLimit(Long appId, String requestUri){
		ConcurrentMap<String,AppUriQpsLimit> uri2qpsLimitMap = appId2uri2qpsLimitMap.get(appId);
		if(uri2qpsLimitMap != null){
			for(Map.Entry<String,AppUriQpsLimit> entry:uri2qpsLimitMap.entrySet()) {
				if(requestUri.startsWith(entry.getKey())){
					AppUriQpsLimit qpsLimit = entry.getValue();
					if (qpsLimit.isExpired()) {
						uri2qpsLimitMap.remove(entry.getKey(), qpsLimit);
						if (uri2qpsLimitMap.isEmpty()) {
							appId2uri2qpsLimitMap.remove(appId);
						}
					} else {
						return qpsLimit;
					}
				}
			}
		}

		return null;
	}

	public void destroy() {
		isDestroyed = true;
        zkClient.unsubscribeAll();
        zkClient.close();
	}

	/**
	 * 限流，展示系统繁忙页面
	 * @param response
	 * @throws IOException
	 */
	private void showBusyPage(HttpServletResponse response) throws IOException {
		response.setContentType("text/html;charset=UTF-8");
		response.setCharacterEncoding("utf-8");
//		JSONObject jsonObject = new JSONObject();
//		jsonObject.put("success", false);
//		jsonObject.put("message", "系统繁忙，请稍后再试");
		String str = "<!DOCTYPE html>\n" +
				"<html xmlns=\"http://www.w3.org/1999/xhtml\">\n" +
				"<head>\n" +
				"<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />\n" +
				"<meta name=\"viewport\" content=\"width=device-width,initial-scale=1.0,minimum-scale=1.0,maximum-scale=1.0,user-scalable=no\" />\n" +
				"<meta name=\"apple-mobile-web-app-capable\" content=\"yes\">\n" +
				"<meta name=\"apple-mobile-web-app-status-bar-style\" content=\"black-translucent\">\n" +
				"<meta content=\"telephone=no\" name=\"format-detection\" />\n" +
				"<title>系统繁忙，请稍后再试</title>\n" +
				"<link rel=\"apple-touch-icon\" href=\"images/app-icon.png\"/>\n" +
				"<link href=\"//yun.duiba.com.cn/webapp/css/404.css\" rel=\"stylesheet\" type=\"text/css\" />\n" +
				"</head>\n" +
				"<body>\n" +
				"<div class=\"content\">\n" +
				"\t<img name=\"\" src=\"//yun.duiba.com.cn/assets/images/error.png\" alt=\"\">\n" +
				"\t<h1>系统繁忙，请稍后再试</h1>\n" +
				"</div>\n" +
				"</body>\n" +
				"</html>\n" +
				"\n";
		PrintWriter out=null;
		try{
			out=response.getWriter();
			out.write(str);
		}finally{
			if(out!=null){
				out.close();
			}
		}
	}


	/**
	 * 限制每个用户每秒最高qps
	 * @param userQpsLimit
	 */
	public void setUserQpsLimit(int userQpsLimit){
		this.userQpsLimit = userQpsLimit;
	}
	public int getUserQpsLimit(){
		return userQpsLimit;
	}

	private static class AppUriQpsLimit{
		private Date finishTime;//临时限流结束时间
		private Integer qpsLimit;//限流的qps值, 为0表示禁止访问
		private String limitUri;//要限流的requestUri路径

		public AppUriQpsLimit() {
		}

		public AppUriQpsLimit(Date finishTime, Integer qpsLimit, String limitUri) {
			this.finishTime = finishTime;
			this.qpsLimit = qpsLimit;
			this.limitUri = limitUri;
		}

		public boolean isExpired(){
			return finishTime.before(new Date());
		}

		public Date getFinishTime() {
			return finishTime;
		}

		public void setFinishTime(Date finishTime) {
			this.finishTime = finishTime;
		}

		public Integer getQpsLimit() {
			return qpsLimit;
		}

		public void setQpsLimit(Integer qpsLimit) {
			this.qpsLimit = qpsLimit;
		}

		public String getLimitUri() {
			return limitUri;
		}

		public void setLimitUri(String limitUri) {
			this.limitUri = limitUri;
		}
	}
//
//	public static void main(String[] args){
//		final int qpsDataCacheSeconds = 5;
//		final LoadingCache<Long,AtomicInteger> consumerId2QpsCache = CacheBuilder.newBuilder()
//				.expireAfterWrite(qpsDataCacheSeconds,TimeUnit.SECONDS)
//				.concurrencyLevel(6)
//				.initialCapacity(10000)
//				.maximumSize(1000000)//最多存100W个consumerId2QpsCache = CacheBuilder.newBuilder()
//                .build(new CacheLoader<Long, AtomicInteger>() {
//                    @Override
//                    public AtomicInteger load(Long key) throws Exception {
//                        return new AtomicInteger(0);
//                    }
//                });
//		ExecutorService es = Executors.newFixedThreadPool(500);
//		List<Runnable> runnables = new ArrayList<>();
//		for(int i=0;i<1000000;i++){
//			final long j = i % 100;
//			runnables.add(new Runnable(){
//
//				@Override
//				public void run() {
//					AtomicInteger qpsAtomic = null;
//					try {
//						qpsAtomic = consumerId2QpsCache.get(j);
//						int qps = qpsAtomic.get();
//						int limit = qpsDataCacheSeconds*USER_QPS_LIMIT;
//						qpsAtomic.incrementAndGet();
//						if(qps>limit){
//							qpsAtomic.incrementAndGet();
//						}
//					} catch (ExecutionException e) {
//						e.printStackTrace();
//					}
//				}
//			});
//		}
//		try {
//			long start = System.currentTimeMillis();
//			ConcurrentUtils.executeTasksBlocking(es, runnables);
//			start = System.currentTimeMillis() - start;
//			System.out.println(start);
//		} catch (InterruptedException e) {
//			e.printStackTrace();
//		}
//
//	}

}
