package cn.com.duibabiz.component.filters.bloom.basic;

import com.google.common.math.DoubleMath;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.util.CollectionUtils;

import java.math.RoundingMode;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * 基于redis布隆过滤器
 * 支持自动扩容
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/4/6 11:50
 */
public class RedisBloomFilter implements ApplicationContextAware {
    public static final Logger LOGGER = LoggerFactory.getLogger(RedisBloomFilter.class);

    private StringRedisTemplate stringRedisTemplate;

    private final BloomFilterConfig bloomFilterConfig;

    public RedisBloomFilter(int growBuffer, int expectedInsertions, double fpp) {
        this.bloomFilterConfig = new BloomFilterConfig(growBuffer, expectedInsertions, fpp);
        LOGGER.info("RedisBloomFilter 初始化完成,数据量={},假阳性概率={},扩容临界buffer={}",
                bloomFilterConfig.getExpectedInsertions(), bloomFilterConfig.getFpp(), bloomFilterConfig.getGrowBuffer());
    }

    /**
     * 获取所有分组key
     *
     * @param key 业务key
     */
    private Set<String> getFilterKeys(String key) {
        return stringRedisTemplate.opsForSet().members(key);
    }

    /**
     * 添加
     *
     * @param expireAt 过期时间，避免泄露,强制每次设置过期时间
     * @return 是否用了新的filterKey
     */
    public boolean put(String key, String value, Date expireAt) {
        String filterKey;
        boolean newKey = false;

        // 获取分组key
        Set<String> filterKeys = getFilterKeys(key);
        if (CollectionUtils.isEmpty(filterKeys)) {
            filterKey = GrowKeyUtil.genFirstKey(key);
            // 添加到集合中
            Long add = stringRedisTemplate.opsForSet().add(key, filterKey);
            if (add != null && add == 1) {
                newKey = true;
            }
            stringRedisTemplate.expireAt(key, expireAt);
        } else {
            filterKey = GrowKeyUtil.findLatest(filterKeys);
        }

        // 判断是否需要扩容
        if (!newKey && needGrow(filterKey)) {
            filterKey = GrowKeyUtil.genNextKey(filterKey);
            Long add = stringRedisTemplate.opsForSet().add(key, filterKey);
            if (add != null && add == 1) {
                newKey = true;
            }
        }

        // hash计算
        int[] offset = bloomFilterConfig.murmurHashOffset(value);
        // 设置位+设置过期时间
        BitMapUtil.putWithPipeline(stringRedisTemplate, filterKey, offset, expireAt);

        return newKey;
    }

    /**
     * 判断是否需要扩容
     */
    private boolean needGrow(String filterKey) {
        long approximateElementCount = approximateElementCount(filterKey);
        float max = bloomFilterConfig.getExpectedInsertions() - bloomFilterConfig.getGrowBuffer();
        boolean needGrow = approximateElementCount > max;
        if (needGrow) {
            LOGGER.info("RedisBloomFilter,需要扩容,filterKey={},当前={},扩容要求={}", filterKey, approximateElementCount, max);
        }
        return needGrow;
    }

    /**
     * 返回已添加到这个Bloom过滤器的不同元素的总数的估计数。
     * 如果这个近似不超过构造过滤器时使用的expectedInsertions的值，那么它是相当准确的。
     */
    private long approximateElementCount(String filterKey) {
        Long bitCount = BitMapUtil.bitCount(stringRedisTemplate, filterKey);

        int bitSize = bloomFilterConfig.getBitSize();
        int numHashFunctions = bloomFilterConfig.getNumHashFunctions();

        double fractionOfBitsSet = (double) bitCount / bitSize;
        return DoubleMath.roundToLong(
                -Math.log1p(-fractionOfBitsSet) * bitSize / numHashFunctions, RoundingMode.HALF_UP);
    }

    /**
     * 返回指定key的扩容信息-数量
     */
    public Map<String, Long> info(String key) {
        Set<String> filterKeys = getFilterKeys(key);
        if (filterKeys == null || filterKeys.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<String, Long> map = new HashMap<>(filterKeys.size() * 2);
        if (!CollectionUtils.isEmpty(filterKeys)) {
            for (String filterKey : filterKeys) {
                map.put(filterKey, approximateElementCount(filterKey));
            }
        }
        return map;
    }

    /**
     * 判断
     * true-可能存在，false-一定不存在
     */
    public boolean mightContain(String key, String value) {
        Set<String> filterKeys = getFilterKeys(key);
        if (CollectionUtils.isEmpty(filterKeys)) {
            return false;
        }

        int[] offset = bloomFilterConfig.murmurHashOffset(value);

        for (String filterKey : filterKeys) {
            // 有一个判断存在则认为存在
            if (BitMapUtil.mightContainWithPipeline(stringRedisTemplate, filterKey, offset)) {
                return true;
            }
        }
        return false;
    }


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.stringRedisTemplate = applicationContext.getBean("stringRedisTemplate", StringRedisTemplate.class);
    }
}