package cn.com.duibabiz.component.filters.bloom.url;

import cn.com.duiba.wolf.entity.Pair;
import cn.com.duibabiz.component.filters.bloom.daily.DailyRedisBloomFilter;
import cn.com.duibabiz.component.filters.bloom.url.handler.PathHandler;
import cn.com.duibabiz.component.filters.bloom.url.handler.PathStrategyFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * 基于布隆过滤器的接口非顺序访问拦截
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/4/12 17:40
 */
public class UrlSerialAccessFilter implements ApplicationContextAware {
    public static final Logger LOGGER = LoggerFactory.getLogger(UrlSerialAccessFilter.class);
    private final DailyRedisBloomFilter dailyRedisBloomFilter;
    private PathStrategyFactory pathStrategyFactory;

    public UrlSerialAccessFilter(DailyRedisBloomFilter dailyRedisBloomFilter) {
        this.dailyRedisBloomFilter = dailyRedisBloomFilter;
    }

    private static String generateValue(Long cid, String path) {
        return cid + "_" + path;
    }

    /**
     * @param cid  用户
     * @param path 当前url
     */
    public void record(Long cid, String path) {
        try {
            PathHandler handler = pathStrategyFactory.getHandler(path);
            if (handler == null) {
                LOGGER.warn("urlSerialAccessService,record,handler is null,path={}", path);
                return;
            }
            String key = handler.generateKey4Record(getHttpServletRequest(), path);
            String value = generateValue(cid, path);
            dailyRedisBloomFilter.put(key, value);
        } catch (Exception e) {
            LOGGER.warn("urlSerialAccessService,record,exception,cid={},path={}", cid, path, e);
        }
    }


    /**
     * 判断是否访问过url
     *
     * @param cid  用户
     * @param path 必须访问的url
     * @return true-可能访问过  false-一定没有访问过
     */
    public Pair<Boolean, String> findFromRecords(Long cid, String path) {
        try {
            PathHandler handler = pathStrategyFactory.getHandler(path);
            if (handler == null) {
                LOGGER.warn("urlSerialAccessService,findFromRecords,handler is null,path={}", path);
                return Pair.from(true, null);
            }
            String key = handler.generateKey4Find(getHttpServletRequest(), path);
            String value = generateValue(cid, path);

            return dailyRedisBloomFilter.mightContain(key, value);
        } catch (Exception e) {
            LOGGER.warn("urlSerialAccessService,findFromRecords,exception,cid={},path={}",
                    cid, path, e);
        }
        return Pair.from(true, null);
    }


    private HttpServletRequest getHttpServletRequest() {
        return ((ServletRequestAttributes) (RequestContextHolder.currentRequestAttributes())).getRequest();
    }


    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.pathStrategyFactory = applicationContext.getBean("pathStrategyFactory", PathStrategyFactory.class);
    }
}
