package cn.com.duiba.biz.tool.duiba.util;

import cn.com.duiba.biz.tool.duiba.client.RequestLocal;
import cn.com.duiba.biz.tool.duiba.dto.RequestLocalParam;
import cn.com.duiba.geo.api.dto.IpAreaDto;
import cn.com.duiba.geo.api.remoteservice.RemoteIpAreaService;
import cn.com.duiba.tuia.union.star.center.api.cache.service.RemoteDeviceUaParseCache;
import cn.com.duiba.tuia.union.star.center.api.remoteservice.domain.rsp.MobileUaInfoDTO;
import cn.com.duiba.wolf.perf.timeprofile.RequestTool;
import cn.com.duiba.wolf.utils.DateUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.env.Environment;

import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * inner日志工具类
 */
public class InnerLogTool {


    private static Environment environment;

    private static String APPLICATION_NAME;

    private static RemoteIpAreaService remoteIpAreaService;

    private static RemoteDeviceUaParseCache remoteDeviceUaParseCache;

    private static final ExecutorService executorService = new ThreadPoolExecutor(30,
            30,
            10, TimeUnit.SECONDS,
            new LinkedBlockingDeque<>(5000),
            new ThreadFactory() {
                int i = 1;

                @Override
                public Thread newThread(@NotNull Runnable r) {
                    return new Thread(r, "InnerLogTool-" + (i++));
                }
            },
            new ThreadPoolExecutor.AbortPolicy());


    private static final Logger logger = LoggerFactory.getLogger("innerLog");

    private static final Logger APPLICATION_LOGGER = LoggerFactory.getLogger(InnerLogTool.class);


    public static Environment environment() {
        if (environment == null) {
            environment = SpringUtil.getBean(Environment.class);
        }
        return environment;
    }

    public static RemoteIpAreaService remoteIpAreaService() {
        if (remoteIpAreaService == null) {
            remoteIpAreaService = SpringUtil.getBean(RemoteIpAreaService.class);
        }
        return remoteIpAreaService;
    }

    public static RemoteDeviceUaParseCache remoteDeviceUaParseCache() {
        if (remoteDeviceUaParseCache == null) {
            remoteDeviceUaParseCache = SpringUtil.getBean(RemoteDeviceUaParseCache.class);
        }
        return remoteDeviceUaParseCache;
    }

    public static ExecutorService executorService() {
        return executorService;
    }

    public static String applicationName() {
        if (StringUtils.isBlank(APPLICATION_NAME)) {
            APPLICATION_NAME = environment().getProperty("spring.application.name");
        }
        return APPLICATION_NAME;
    }

    /**
     * 构建基础信息
     * @param log 日志内容
     * @param type 埋点类型
     * @return 日志json内容
     */
    private static JSONObject buildBasicInfo(JSONObject log, int type) {
        JSONObject json = new JSONObject();
        log.put("group", 2);
        log.put("type", type);
        log.put("time", DateUtils.getSecondStr(new Date()));
        log.put("json", json);
        return json;
    }

    /**
     * 构建请求信息
     * @param json 日志json内容
     */
    private static void buildRequestInfo(JSONObject json, RequestLocalParam requestLocalMessage) {
        json.put("consumerId", requestLocalMessage.getConsumerId());
        json.put("appId", requestLocalMessage.getAppId());
        String ip = requestLocalMessage.getIp();
        json.put("ip", ip);
        if (StringUtils.isNotBlank(ip)) {
            IpAreaDto ipInfo = getIpInfo(ip);
            if (ipInfo != null) {
                json.put("country", ipInfo.getCountry());
                json.put("province", ipInfo.getRegion());
                json.put("city", ipInfo.getCity());
            }
        }
        String userAgent = requestLocalMessage.getUserAgent();
        json.put("ua", userAgent);
        json.put("os", RequestTool.getOS(json.getString("ua")));
    }


    /**
     * 构建系统信息
     * @param json 日志json内容
     */
    private static void buildSysInfo(JSONObject json) {
        json.put("duibaPlatform", applicationName());
    }

    /**
     * 构建uaInfo信息
     * @param extraInfo extraInfo
     * @param json 日志json内容
     */
    private static void buildExtraInfo(String extraInfo, JSONObject json) {
        if (StringUtils.isNotBlank(extraInfo)) {
            json.putAll(JSONObject.parseObject(extraInfo));
        }
    }

    /**
     * 构建uaInfo信息
     * @param requestLocalParam 请求相关
     * @param json 日志json内容
     */
    public static void buildUaInfo(RequestLocalParam requestLocalParam, JSONObject json) {
        if (json != null) {
            String userAgent = requestLocalParam.getUserAgent();
            if (StringUtils.isNotBlank(userAgent)) {
                MobileUaInfoDTO uaInfo = getUAInfo(userAgent);
                if (uaInfo != null) {
                    json.putAll(JSONObject.parseObject(JSON.toJSONString(uaInfo)));
                }
            }
        }
    }

    /**
     * 构建默认的请求信息（从RequestLocal获取）
     * @return 请求信息
     */
    public static RequestLocalParam buildDefaultRequestLocalMessage() {
        RequestLocalParam requestLocalParam = new RequestLocalParam();
        requestLocalParam.setIp(RequestLocal.getIp());
        requestLocalParam.setConsumerId(RequestLocal.getCid());
        requestLocalParam.setAppId(RequestLocal.getAppId());
        requestLocalParam.setUserAgent(RequestLocal.getUserAgent());
        return requestLocalParam;
    }


    /**
     * 参与日志
     * @param requestLocalParam  请求相关
     * @param logType 日志类型
     * @param activityType  活动类型
     * @param activityId    入库ID
     * @param id    活动通用表id
     * @param extraInfo 扩展信息
     */
    public static void innerLog(RequestLocalParam requestLocalParam, Integer logType, Integer activityType, Long id, Long activityId, String extraInfo) {
        executorService().execute(() -> {
            JSONObject log = new JSONObject();
            JSONObject json = buildBasicInfo(log, logType);
            if (activityType != null) {
                json.put("activityType", activityType);
            }
            if (activityType != null) {
                json.put("activityId", activityId);
            }
            if (id != null) {
                json.put("id", id);
            }
            buildRequestInfo(json, requestLocalParam);
            buildUaInfo(requestLocalParam, json);
            buildSysInfo(json);
            buildExtraInfo(extraInfo, json);
            logger.info(log.toJSONString());
        });
    }

    /**
     * 获取ipInfo信息
     * @param ip ip
     * @return ipInfo
     */
    public static IpAreaDto getIpInfo(String ip) {
        IpAreaDto ipInfo = null;
        try {
            ipInfo = remoteIpAreaService().findIpInfo(ip);
        } catch (Exception e) {
            APPLICATION_LOGGER.warn("ip解析失败", e);
        }
        return ipInfo;
    }

    /**
     * 获取uaInfo信息
     * @param ua ua
     * @return uaInfo
     */
    public static MobileUaInfoDTO getUAInfo(String ua) {
        MobileUaInfoDTO mobileInfoByUa = null;

        if (StringUtils.isBlank(ua)) {
            return null;
        }
        try {
            if (StringUtils.isNoneBlank(ua)) {
                mobileInfoByUa = remoteDeviceUaParseCache().getMobileInfoByUa(ua);
            }
        } catch (Exception e) {
            APPLICATION_LOGGER.warn("ua解析失败", e);
        }
        return mobileInfoByUa;
    }


}
