package cn.com.duibabiz.component.filters.bloom.url;

import cn.com.duiba.wolf.entity.Pair;
import cn.com.duibabiz.component.filters.bloom.daily.DailyRedisBloomFilter;
import cn.com.duibabiz.component.filters.bloom.url.path.PathParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;

/**
 * 基于布隆过滤器的接口非顺序访问拦截
 *
 * @author zsp (zengshuiping@duiba.com.cn)
 * @date 2021/4/12 17:40
 */
public class UrlSerialAccessFilter {
    public static final Logger LOGGER = LoggerFactory.getLogger(UrlSerialAccessFilter.class);
    private final DailyRedisBloomFilter dailyRedisBloomFilter;

    public UrlSerialAccessFilter(DailyRedisBloomFilter dailyRedisBloomFilter) {
        this.dailyRedisBloomFilter = dailyRedisBloomFilter;
    }

    /**
     * 工具方法，获取HttpServletRequest 线程安全
     */
    private static HttpServletRequest getHttpServletRequest() {
        return ((ServletRequestAttributes) (RequestContextHolder.currentRequestAttributes())).getRequest();
    }

    private static String generateValue(Long cid, String path) {
        return cid + "_" + path;
    }

    /**
     * 记录
     */
    public void record(PathParser pathParser, Long cid) {
        if (pathParser == null || cid == null || cid <= 0L) {
            return;
        }
        String path = pathParser.getPath();
        try {
            // 根据请求生成key
            String key = pathParser.getParser().parse(getHttpServletRequest());
            String value = generateValue(cid, path);
            dailyRedisBloomFilter.put(key, value);
        } catch (Exception e) {
            LOGGER.warn("urlSerialAccessService,record,exception,cid={},path={}", cid, path, e);
        }
    }

    /**
     * 是否有记录
     * key：true已访问，false 未访问  value:path
     */
    public Pair<Boolean, String> hasRecord(PathParser pathParser, Long cid, String targetPath) {
        Pair<Boolean, String> pair = Pair.from(true, null);
        if (pathParser == null || cid == null || cid <= 0L) {
            return pair;
        }
        String path = pathParser.getPath();
        try {
            // 根据请求生成key
            String key = pathParser.getParser().parse(getHttpServletRequest());
            String value = generateValue(cid, targetPath);
            Pair<Boolean, String> mightContain = dailyRedisBloomFilter.mightContain(key, value);
            return Pair.from(mightContain.getKey(), path);
        } catch (Exception e) {
            LOGGER.warn("urlSerialAccessService,findFromRecords,exception,cid={},path={}",
                    cid, path, e);
        }
        return pair;
    }
}
