package cn.com.duiba.boot.perftest;

import org.apache.commons.lang3.StringUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;

import org.springframework.context.ApplicationContext;

public final class PerfTestUtils {
    private PerfTestUtils(){}

    public static final String PERF_TEST_KEY = "_duibaPerf";
    public static final String PERF_TEST_SCENE_ID_KEY = "_duibaPerfSceneId";
    public static final String PERF_TEST_CLUSTER = "_duibaPerfTestCluster";

    public static final String IS_PERF_TEST_MODE_HEADER_KEY = "isPerfTestMode";

    //压测场景id，如果eureka注册信息中包含此字段，表示这个容器只用于线上压测，正常流量不应该调度到这个容器，而且不同场景id之间流量也要隔离(如果这个key设置在系统环境变量中，则key为 DUIBA_PERF_SCENE_ID)
    //详情见此文档：  http://cf.dui88.com/pages/viewpage.action?pageId=10869508
    public static final String DUIBA_PERF_SCENE_ID = "duiba.perf.scene.id";

    /**
     * 检测到包含_duibaPerf=1参数的url(或cookie中有_duibaPerf=1,或header中包含_duibaPerf=1)则认为是压测请求
     * @param request
     * @return
     */
    public static boolean isPerfTestRequest(HttpServletRequest request){
        boolean isTestMode = isPerfTestFromParameter(request);

        if(!isTestMode){
            String perfTestHeader = request.getHeader(PERF_TEST_KEY);
            if(perfTestHeader != null && ("1".equals(perfTestHeader) || "true".equals(perfTestHeader))){
                isTestMode = true;
            }
        }

        if(!isTestMode && "true".equals(request.getHeader(IS_PERF_TEST_MODE_HEADER_KEY))){
            //feign 发起的http压测请求头中会带有isPerfTestMode=true
            isTestMode = true;
        }

        return isTestMode;
    }

    /**
     * 从request中获取压测场景id，压测场景ID可以在以下任一地方：
     * <br/>
     * 参数的url中带有 _duibaPerfSceneId=**
     * <br/>
     * cookie中有 _duibaPerfSceneId=**
     * <br/>
     * header中包含 _duibaPerfSceneId=**
     *
     * @param request
     * @return 压测场景ID，可能为null (如果没有压测标记，无论有没有压测场景ID，都返回null)
     */
    public static String getPerfTestSceneId(HttpServletRequest request){
        if(!isPerfTestRequest(request)){//如果没有压测标记，无论有没有压测场景ID，都返回null
            return null;
        }

        String sceneId = request.getParameter(PERF_TEST_SCENE_ID_KEY);

        if(StringUtils.isBlank(sceneId)){
            Cookie[] cookies = request.getCookies();
            if(cookies != null) {
                for (Cookie cookie : cookies) {
                    if(PERF_TEST_SCENE_ID_KEY.equals(cookie.getName())){
                        sceneId = cookie.getValue();
                    }
                }
            }
        }
        if(StringUtils.isBlank(sceneId)){
            sceneId = request.getHeader(PERF_TEST_SCENE_ID_KEY);
        }

        return sceneId;
    }

    /**
     * 从 request 中获取压测集群标识，压测集群标识可以再以下任一地方
     * <br/>
     * 参数的url中带有 _duibaPerfTestCluster= 1 or 0
     * <br/>
     * cookie中有 _duibaPerfTestCluster= 1 or 0
     * <br/>
     * header中包含 _duibaPerfTestCluster= 1 or 0
     * @param request
     * @return 当且仅当 _duibaPerfTestCluster = 1 or true 的情况下，返回true，其余时候都是 false
     */
    public static boolean isPerfTestCluster(HttpServletRequest request) {
        if (!isPerfTestRequest(request)) { // 如果没有压测标记，无论有没有压测集群标识，都返回 false
            return false;
        }

        String isTestCluster = request.getParameter(PERF_TEST_CLUSTER);

        if (StringUtils.isBlank(isTestCluster)) {
            Cookie[] cookies = request.getCookies();
            if (cookies != null) {
                for (Cookie cookie : cookies) {
                    if (PERF_TEST_CLUSTER.equals(cookie.getName())) {
                        isTestCluster = cookie.getValue();
                    }
                }
            }
        }

        if (StringUtils.isBlank(isTestCluster)) {
            isTestCluster = request.getHeader(PERF_TEST_CLUSTER);
        }

        return "1".equals(isTestCluster) || "true".equals(isTestCluster);
    }

    private static boolean isPerfTestFromParameter(HttpServletRequest request){
        boolean isTestMode = false;
        String testInParameter = request.getParameter(PERF_TEST_KEY);
        if(testInParameter != null && ("1".equals(testInParameter) || "true".equals(testInParameter))){
            isTestMode = true;
        }else{
            Cookie[] cookies = request.getCookies();
            if(cookies != null) {
                for (Cookie cookie : cookies) {
                    if(PERF_TEST_KEY.equals(cookie.getName()) && ("1".equals(cookie.getValue()) || "true".equals(cookie.getValue()))){
                        isTestMode = true;
                    }
                }
            }
        }

        return isTestMode;
    }

    /**
     * 获得当前机器实例的独立容器实例压测场景ID
     * @param applicationContext
     * @return
     */
    public static String getThisServerDuibaPerfSceneId(ApplicationContext applicationContext){
        return applicationContext.getEnvironment().getProperty(DUIBA_PERF_SCENE_ID);
    }
}
