package cn.com.duiba.boot.ext.autoconfigure.graceclose;

import com.alibaba.dubbo.config.ProtocolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.SpringApplicationRunListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;

import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;

/**
 * 自动配置优雅停机
 * <br/>
 *
 * 不能优雅停机的原因：各个框架包括spring、dubbo、xmemcached等在初始化时都添加了自己的shutdownhook，而且这些shutdownhook的执行是无序的，所以有可能在dubbo销毁前xmemcached、datasource等已经销毁，导致关闭期间的几秒内大量报错
 * <br/>
 *
 * 解决方法：在spring启动后通过反射移除所有shutdownhook，并添加必要的spring的shutdownhook、添加必要的dubbo清理代码，实现优雅停机
 *
 * Created by wenqi.huang on 2016/12/27.
 */
public class GracefulCloseRunListener implements SpringApplicationRunListener {

    private static final Logger logger = LoggerFactory.getLogger(GracefulCloseRunListener.class);

    private SpringApplication application;

    /**
     * 必须有这个构造函数，否则spring无法初始化该类
     * @param application
     * @param args
     */
    public GracefulCloseRunListener(SpringApplication application, String[] args) {
        this.application = application;
    }

    @Override
    public void started() {

    }

    @Override
    public void environmentPrepared(ConfigurableEnvironment environment) {

    }

    @Override
    public void contextPrepared(ConfigurableApplicationContext context) {

    }

    @Override
    public void contextLoaded(ConfigurableApplicationContext context) {

    }

    @Override
    public void finished(final ConfigurableApplicationContext context, Throwable exception) {
        if(context == null){
            return;
        }
        try {
            Class clazz = Class.forName("java.lang.ApplicationShutdownHooks");
            Field field = clazz.getDeclaredField("hooks");
            field.setAccessible(true);
            IdentityHashMap<Thread, Thread> hooks = (IdentityHashMap<Thread, Thread>) field.get(null);

            //清空原有hook,但保留日志的hook
            List<Thread> keys2remove = new ArrayList<>();
            for(Map.Entry<Thread, Thread> entry : hooks.entrySet()){
                String hookClassName = entry.getKey().getClass().getName();
                if(hookClassName.equals("java.util.logging.LogManager$Cleaner")
                        || hookClassName.equals("org.unidal.helper.Threads$Manager$1") //cat
                        || hookClassName.startsWith("org.jacoco.agent.rt")){ //jacoco,去掉shutdownhook这个会导致sonarqube 任务执行失败
                    continue;
                }
                logger.debug("remove shotdownhook:{}", hookClassName);
                keys2remove.add(entry.getKey());
            }
            for(Thread t : keys2remove) {
                hooks.remove(t);
            }

            //添加自己的清理hook
            Runtime.getRuntime().addShutdownHook(new Thread() {
                @Override
                public void run() {
                    long start = System.currentTimeMillis();
                    try {
                        Class.forName("com.alibaba.dubbo.config.ProtocolConfig");
                        ProtocolConfig.destroyAll();//先清理dubbo
                    } catch(ClassNotFoundException e){
                        logger.info("该项目没有引入dubbo，不会执行dubbo清理");
                        // Ignore
                    } catch(ExceptionInInitializerError e){
                        logger.info("该项目没有使用dubbo，不会执行dubbo清理");
                        // Ignore
                    } catch (Throwable e) {
                        logger.error(e.getMessage(), e);
                    }

                    try {
                        context.close();//再清理spring
                    } catch (Throwable e) {
                        logger.error(e.getMessage(), e);
                    }
                    long period = System.currentTimeMillis() - start;
                    logger.debug("执行shutdownhook成功，cost：{} ms", period);
                }
            });
        } catch (IllegalStateException e) {
            // If the VM is already shutting down,
            // We do not need to register shutdownHook.
        } catch (Exception e){
            logger.error(e.getMessage(), e);
        }
    }

}
