package cn.com.duiba.boot.ext.autoconfigure.ons;

import com.aliyun.openservices.ons.api.*;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import javax.annotation.Resource;
import java.util.Properties;

/**
 * aliyun ONS消息队列自动配置
 * Created by wenqi.huang on 2016/12/26.
 */
@Configuration
@EnableConfigurationProperties(OnsProperties.class)
public class OnsAutoConfiguration {

    protected static abstract class OnsClientConfiguration{

        @Autowired
        protected OnsProperties onsProperties;

    }

    @Configuration
    @ConditionalOnClass({Producer.class})
    @ConditionalOnMissingBean(name={"onsProducer"})
    @ConditionalOnProperty(name="duiba.ons.producer.enable", havingValue = "true", matchIfMissing = false)
    protected static class OnsProducerConfigurator extends OnsClientConfiguration{

        @Bean(name="onsProducer", destroyMethod = "shutdown")
        public Producer onsProducer(){
            Properties properties = new Properties();
            properties.put(PropertyKeyConst.ProducerId, "PID-hwq-trade-center");// 您在MQ控制台创建的Producer ID
            properties.put(PropertyKeyConst.AccessKey, onsProperties.getAccessKey());// 鉴权用AccessKey
            properties.put(PropertyKeyConst.SecretKey, onsProperties.getSecretKey());// 鉴权用SecretKey
            properties.put(PropertyKeyConst.SendMsgTimeoutMillis, onsProperties.getProducer().getSendMsgTimeoutMillis());// 设置发送超时时间，单位毫秒
            //公有云生产环境：http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //公有云公测环境：http://onsaddr-internet.aliyun.com/rocketmq/nsaddr4client-internet
            //杭州金融云环境：http://jbponsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //杭州深圳云环境：http://mq4finance-sz.addr.aliyun.com:8080/rocketmq/nsaddr4client-internal
//            properties.put(PropertyKeyConst.ONSAddr,
//                    "http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal");//此处以公有云生产环境为例
            Producer producer = ONSFactory.createProducer(properties);
            // 在发送消息前，必须调用start方法来启动Producer，只需调用一次即可
            producer.start();

            return producer;
        }
    }

    @Configuration
    @ConditionalOnClass({Consumer.class})
    @ConditionalOnMissingBean(name={"onsConsumer"})
    @ConditionalOnProperty(name="duiba.ons.consumer.enable", havingValue = "true", matchIfMissing = false)
    protected static class OnsConsumerConfigurator extends OnsClientConfiguration{

        @Bean(name="onsConsumer")
        public FactoryBean<Consumer> onsConsumer() throws ClassNotFoundException, IllegalAccessException, InstantiationException {
            return new OnsConsumerFactoryBean(onsProperties);
        }
    }

    private static class OnsConsumerFactoryBean implements FactoryBean<Consumer>,InitializingBean{

        @Resource(name="bootOnsMessageListener",type = MessageListener.class)
        private MessageListener onsMessageListener;

        private Consumer onsConsumer;
        private OnsProperties onsProperties;

        public OnsConsumerFactoryBean(OnsProperties onsProperties){
            this.onsProperties = onsProperties;
        }

        @Override
        public void afterPropertiesSet() throws Exception {
            Properties properties = new Properties();
            properties.put(PropertyKeyConst.ConsumerId, onsProperties.getConsumer().getGroup());// 您在MQ控制台创建的Consumer ID
            properties.put(PropertyKeyConst.AccessKey, onsProperties.getAccessKey());// 鉴权用AccessKey，在阿里云服务器管理控制台创建
            properties.put(PropertyKeyConst.SecretKey, onsProperties.getSecretKey());// 鉴权用SecretKey，在阿里云服务器管理控制台创建
            properties.put(PropertyKeyConst.MaxReconsumeTimes, onsProperties.getConsumer().getMaxReconsumeTimes());// 消息消费失败时的最大重试次数,默认值16

            if(onsProperties.getConsumer().getConsumeThreadNums() > 0) {
                properties.put(PropertyKeyConst.ConsumeThreadNums, onsProperties.getConsumer().getConsumeThreadNums());// 消费线程数量
            }

            //公有云生产环境：http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //公有云公测环境：http://onsaddr-internet.aliyun.com/rocketmq/nsaddr4client-internet
            //杭州金融云环境：http://jbponsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //杭州深圳云环境：http://mq4finance-sz.addr.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //properties.put(PropertyKeyConst.ONSAddr, "http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal");//此处以公有云生产环境为例
            com.aliyun.openservices.ons.api.Consumer consumer = ONSFactory.createConsumer(properties);
            String topics = onsProperties.getConsumer().getTopics();
            String[] topicArr = topics.split(",");

            for(String topic : topicArr) {
                consumer.subscribe(topic, "*", onsMessageListener);
            }

            consumer.start();
            onsConsumer = consumer;
        }

        @Override
        public Consumer getObject() throws Exception {
            return onsConsumer;
        }

        @Override
        public Class<?> getObjectType() {
            return Consumer.class;
        }

        @Override
        public boolean isSingleton() {
            return true;
        }
    }
}
