package cn.com.duiba.boot.ext.autoconfigure.initserver;

import java.sql.Connection;
import java.sql.SQLException;

import com.alibaba.dubbo.config.spring.ReferenceBean;
import org.apache.commons.dbcp2.BasicDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.embedded.ServletRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import com.alibaba.dubbo.rpc.service.EchoService;
import org.springframework.core.Ordered;
import org.springframework.jdbc.CannotGetJdbcConnectionException;

/**
 * 应用启动时初始化，防止第一次请求执行慢
 */
@Configuration
public class InitServerAutoConfiguration {
	private static Logger logger = LoggerFactory.getLogger(InitServerAutoConfiguration.class);

	/**
	 * 初始化Servlet
	 */
	@Configuration
	@ConditionalOnWebApplication
	@ConditionalOnClass({ServletRegistrationBean.class})
	public static class WebPostProcessorConfiguration{
		
		@Bean
		public BeanPostProcessor webPostProcessorConfigurer(){
			return new BeanPostProcessor() {
				
				@Override
				public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
					return bean;
				}

				@Override
				public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
					if(bean instanceof ServletRegistrationBean){
						//启动时初始化 dispatcherServlet
						ServletRegistrationBean servlet = (ServletRegistrationBean)bean;
						if(servlet.getServletName().equalsIgnoreCase("dispatcherServlet")){
							servlet.setLoadOnStartup(1);
						}
					}
					return bean;
				}
			};
		}
	}

	/**
	 * 初始化Dubbo
	 */
	@Configuration
	@ConditionalOnClass({EchoService.class})
	public static class DubboPostProcessorConfiguration{
		
		@Bean
		public BeanPostProcessor dubboPostProcessorConfigurer(){
			return new BeanPostProcessor() {
				
				@Override
				public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
					return bean;
				}

				@Override
				public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
					if(bean instanceof EchoService){
						//dubbo回声检测，用于检测dubbo调用链路是否通畅 （即初始化dubbo）
						try {
							EchoService echoService = (EchoService)bean;
							echoService.$echo("OK");
						} catch (Exception e) {
							logger.error(e.getMessage(), e);
						}
					}
					return bean;
				}
			};
		}
	}

	/**
	 * 初始化jdbc
	 */
	@Configuration
	@ConditionalOnClass({BasicDataSource.class})
	public static class DataSourcePostProcessorConfiguration{
		
		@Bean
		public BeanPostProcessor dataSourcePostProcessorConfigurer(){
			return new BeanPostProcessor() {
				
				@Override
				public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
					return bean;
				}

				@Override
				public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
					if(bean instanceof BasicDataSource){
						//初始化数据库连接池
						Connection conn = null;
						try {
							BasicDataSource dataSource = (BasicDataSource)bean;
							conn = dataSource.getConnection();
						} catch (SQLException e) {
							throw new CannotGetJdbcConnectionException("Could not get JDBC Connection", e);
						} finally{
							if(conn != null){
								try {
									conn.close();
								} catch (SQLException e) {}
							}
						}
					}
					return bean;
				}
			};
		}
	}

	/**
	 * 写点注释,迁移到dubbo里（这个方法废弃）
	 */
//	@Configuration
//	@ConditionalOnClass({ReferenceBean.class})
//	public static class DubboPostFactoryProcessorConfiguration{
//
//		@Bean
//		public static BeanFactoryPostProcessor dubboBeanFactoryPostProcessorConfigurer(){
//			return new DubboBeanFactoryPostProcessor();
//		}
//	}
//
//	private static class DubboBeanFactoryPostProcessor implements Ordered, BeanFactoryPostProcessor {
//		@Override
//		public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
//			for(String beanName : beanFactory.getBeanDefinitionNames()){
//				BeanDefinition bd = beanFactory.getBeanDefinition(beanName);
//				if(bd.isSingleton() && !bd.isAbstract() && !bd.isLazyInit() && ReferenceBean.class.getName().equals(bd.getBeanClassName())){
//					try {
//						beanFactory.getBean(beanName);
//					}catch(NoSuchBeanDefinitionException e){
//						// Ignore
//					}
//				}
//			}
//		}
//
//		@Override
//		public int getOrder() {
//			return Ordered.LOWEST_PRECEDENCE;
//		}
//	}
}
