package cn.com.duiba.boot.ext.autoconfigure.web;

import org.apache.catalina.connector.ClientAbortException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.WebMvcAutoConfiguration;
import org.springframework.boot.context.embedded.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.web.accept.ContentNegotiationManager;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.view.ContentNegotiatingViewResolver;

import javax.servlet.DispatcherType;
import javax.servlet.Servlet;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

/**
 * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
 * <br/>
 * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
 * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
 * <br/>
 * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
 * Created by wenqi.huang on 2017/1/5.
 */
@Configuration
@AutoConfigureBefore({WebMvcAutoConfiguration.class})
@ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
        WebMvcConfigurerAdapter.class })
@ConditionalOnWebApplication
public class WebMvcFixAutoConfiguration {

    /**
     * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
     * <br/>
     * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
     * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
     * <br/>
     * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
     */
    @Bean(name = "viewResolver")
    @ConditionalOnMissingBean(name = "viewResolver", value = ContentNegotiatingViewResolver.class)
    public ContentNegotiatingViewResolver viewResolver(BeanFactory beanFactory) {
        ContentNegotiatingViewResolver resolver = new ContentNegotiatingViewResolver();
        resolver.setContentNegotiationManager(
                beanFactory.getBean(ContentNegotiationManager.class));
        // ContentNegotiatingViewResolver uses all the other view resolvers to locate
        // a view so it should have a high precedence
        resolver.setOrder(Ordered.LOWEST_PRECEDENCE);
        return resolver;
    }

    /**
     * 自动加入Http Filter，对超时请求打印warn日志
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
            WebMvcConfigurerAdapter.class })
    @ConditionalOnWebApplication
    public static class dbTimeProfileFilterConfiguration{
        @Bean
        public FilterRegistrationBean dbTimeProfileFilterConfigurer(){
            DBTimeProfileFilter filter=new DBTimeProfileFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<String>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(-1);
            return registrationBean;
        }
    }

    /**
     * 自动加入Http Filter，拦截这个异常：org.apache.catalina.connector.ClientAbortException: java.io.IOException: 断开的管道
     * <br/>
     * 改为打印warn日志
     * 避免其上报给loginsight和cat监控。
     */
    @Configuration
    @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
            WebMvcConfigurerAdapter.class, ClientAbortException.class })
    @ConditionalOnWebApplication
    public static class ClientAbortExceptionIgnoreFilterConfiguration{
        @Bean
        public FilterRegistrationBean clientAbortExceptionIgnoreFilterConfigurer(){
            ClientAbortExceptionIgnoreFilter filter = new ClientAbortExceptionIgnoreFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<String>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(2);//在cat后面，防止不必要的异常让cat上报
            return registrationBean;
        }
    }

}
