package cn.com.duiba.boot.ext.autoconfigure.initserver;

import cn.com.duiba.boot.ext.autoconfigure.core.AsyncSpecifiedBeanPostProcessor;
import cn.com.duiba.boot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * Created by wenqi.huang on 2017/6/16.
 */
public class ApplicationListenerForAsyncSpecified implements ApplicationListener<ContextRefreshedEvent>, Ordered {
    private List<AsyncSpecifiedBeanPostProcessor> asyncProcessorList;
    private ApplicationContext applicationContext;

    public ApplicationListenerForAsyncSpecified(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
        //找到所有SpecifiedBeanPostProcessor实例并排序，然后在BeanPostProcessor中按顺序处理
        Map<String,AsyncSpecifiedBeanPostProcessor> asyncProcessorMap = applicationContext.getBeansOfType(AsyncSpecifiedBeanPostProcessor.class, false, false);
        List<AsyncSpecifiedBeanPostProcessor> asyncProcessorList = new ArrayList<>(asyncProcessorMap.values());

        this.asyncProcessorList = asyncProcessorList;
    }

    //指定顺序，让当前的ApplicationListener被优先执行（比Dubbo暴露服务先执行）
    @Override
    public int getOrder() {
        return -2;
    }

    @Override
    public void onApplicationEvent(ContextRefreshedEvent event) {
        //开启线程池用于异步方法，执行完成后立即销毁
        ExecutorService executorService = Executors.newFixedThreadPool(100, new NamedThreadFactory("DuibaBootInitServerThread-"));
        //异步化执行所有AsyncSpecifiedBeanPostProcessor
        List<Map<String, Object>> beans2Process = new ArrayList<>();
        int totalBean2ProcessCount = 0;
        for(AsyncSpecifiedBeanPostProcessor p : asyncProcessorList){
            Map<String, Object> beanMap = applicationContext.getBeansOfType(p.getBeanType(), false,false);
            beans2Process.add(beanMap);
            totalBean2ProcessCount += beanMap.size();
        }

        final CountDownLatch countDownLatch = new CountDownLatch(totalBean2ProcessCount);
        int i = 0;
        for(Map<String, Object> beanMap : beans2Process){
            final AsyncSpecifiedBeanPostProcessor processor = asyncProcessorList.get(i++);
            for(final Map.Entry<String, Object> entry : beanMap.entrySet()){
                executorService.submit(new Runnable() {
                    @Override
                    public void run() {
                        processor.postProcessAfterInitialization(entry.getValue(), entry.getKey());
                        countDownLatch.countDown();
                    }
                });
            }
        }

        //阻塞住spring流程，确保异步初始化全部执行完成再继续
        try {
            countDownLatch.await();
        }catch(InterruptedException e){
            //Ignore
        }

        executorService.shutdown();
    }
}
