package cn.com.duiba.boot.ext.autoconfigure.initserver;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.*;

import cn.com.duiba.boot.ext.autoconfigure.core.AsyncSpecifiedBeanPostProcessor;
import cn.com.duiba.boot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.apache.commons.dbcp2.BasicDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.context.embedded.ServletRegistrationBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import com.alibaba.dubbo.rpc.service.EchoService;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.dao.DataAccessException;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.CannotGetJdbcConnectionException;
import redis.clients.jedis.Jedis;

/**
 * 应用启动时初始化，防止第一次请求执行慢
 */
@Configuration
public class InitServerAutoConfiguration {
	private static Logger logger = LoggerFactory.getLogger(InitServerAutoConfiguration.class);

//	private Map<Class<?>, List<?>> clazz2BeansMap = new HashMap<>(10);

	/**
	 * 用于管理所有的SpecifiedBeanPostProcessor，对他们进行调用
	 */
	@Bean
	public BeanPostProcessor specifiedBeanPostProcessorConfigurer(ApplicationContext applicationContext){
		//找到所有SpecifiedBeanPostProcessor实例并排序，然后在BeanPostProcessor中按顺序处理
		Map<String,SpecifiedBeanPostProcessor> processorMap = applicationContext.getBeansOfType(SpecifiedBeanPostProcessor.class, false, false);
		List<SpecifiedBeanPostProcessor> processorList = new ArrayList<>(processorMap.values());
		Collections.sort(processorList, new Comparator<SpecifiedBeanPostProcessor>() {
			@Override
			public int compare(SpecifiedBeanPostProcessor o1, SpecifiedBeanPostProcessor o2) {
				return o1.getOrder() - o2.getOrder();
			}
		});

		return new BeanPostProcessorOfSpecified(processorList);
	}

	/**
	 * 用于管理所有的AsyncSpecifiedBeanPostProcessor，对他们进行调用,这个Listener会阻塞住spring流程，确保异步初始化全部执行完成再继续
	 */
	@Bean
	public ApplicationListener<ContextRefreshedEvent> asyncSpecifiedBeanPostProcessorConfigurer(ApplicationContext applicationContext){
		//找到所有SpecifiedBeanPostProcessor实例并排序，然后在ApplicationListener中按顺序处理
		return new ApplicationListenerForAsyncSpecified(applicationContext);
	}

	/**
	 * 初始化Servlet
	 */
	@Configuration
	@ConditionalOnWebApplication
	@ConditionalOnClass({ServletRegistrationBean.class})
	public static class WebPostProcessorConfiguration{
		
		@Bean
		public SpecifiedBeanPostProcessor webPostProcessorConfigurer(){
			return new SpecifiedBeanPostProcessor<ServletRegistrationBean>() {

				@Override
				public int getOrder() {
					return 0;
				}

				@Override
				public Class<ServletRegistrationBean> getBeanType() {
					return ServletRegistrationBean.class;
				}

				@Override
				public Object postProcessBeforeInitialization(ServletRegistrationBean bean, String beanName) throws BeansException {
					return bean;
				}

				@Override
				public Object postProcessAfterInitialization(ServletRegistrationBean bean, String beanName) throws BeansException {
					//启动时初始化 dispatcherServlet
					if(bean.getServletName().equalsIgnoreCase("dispatcherServlet")){
						bean.setLoadOnStartup(1);
					}
					return bean;
				}
			};
		}
	}

	/**
	 * 并发初始化Dubbo
	 */
	@Configuration
	@ConditionalOnClass({EchoService.class})
	public static class DubboPostProcessorConfiguration{
		
		@Bean
		public AsyncSpecifiedBeanPostProcessor dubboPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<EchoService>() {

				@Override
				public Class<EchoService> getBeanType() {
					return EchoService.class;
				}

				@Override
				public void postProcessAfterInitialization(EchoService echoService, String beanName) {
					//dubbo回声检测，用于检测dubbo调用链路是否通畅 （即初始化dubbo）
					try {
						echoService.$echo("OK");
					} catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
				}
			};
		}
	}

	/**
	 * 初始化jdbc
	 */
	@Configuration
	@ConditionalOnClass({BasicDataSource.class})
	public static class DataSourcePostProcessorConfiguration{
		
		@Bean
		public AsyncSpecifiedBeanPostProcessor dataSourcePostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<BasicDataSource>() {

				@Override
				public Class<BasicDataSource> getBeanType() {
					return BasicDataSource.class;
				}
				
				@Override
				public void postProcessAfterInitialization(BasicDataSource dataSource, String beanName) {
					//初始化数据库连接池
					Connection conn = null;
					try {
						conn = dataSource.getConnection();
					} catch (SQLException e) {
						throw new CannotGetJdbcConnectionException("Could not get JDBC Connection", e);
					} finally{
						if(conn != null){
							try {
								conn.close();
							} catch (SQLException e) {}
						}
					}
				}
			};
		}
	}

	/**
	 * 初始化redis
	 */
	@Configuration
	@ConditionalOnClass({RedisTemplate.class})
	public static class RedisPostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor redisPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<RedisTemplate>() {

				@Override
				public Class<RedisTemplate> getBeanType() {
					return RedisTemplate.class;
				}

				@Override
				public void postProcessAfterInitialization(RedisTemplate redisTemplate, String beanName) {
					//调用redis初始化，由于第一次redis调用会比较慢（尤其是其中Cat监控会给redisTemplate加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量redis调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					redisTemplate.execute(new RedisCallback() {
						@Override
						public Object doInRedis(RedisConnection connection) throws DataAccessException {
							Jedis jedis = (Jedis) connection.getNativeConnection();
							return jedis.get("justForInitTest");
						}
					});
				}
			};
		}
	}

	/**
	 * 初始化elasticSearch
	 */
	@Configuration
	@ConditionalOnClass({ElasticsearchOperations.class})
	public static class ElasticSearchPostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor elasticSearchPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<ElasticsearchOperations>() {

				@Override
				public Class<ElasticsearchOperations> getBeanType() {
					return ElasticsearchOperations.class;
				}

				@Override
				public void postProcessAfterInitialization(ElasticsearchOperations elasticsearchOperations, String beanName) {
					//调用elasticSearch初始化，由于第一次elasticSearch调用会比较慢（尤其是其中Cat监控会给elasticSearch加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量elasticSearch调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					elasticsearchOperations.indexExists("justForTest");
				}
			};
		}
	}

	/**
	 * 初始化MongoDb
	 */
	@Configuration
	@ConditionalOnClass({MongoOperations.class})
	public static class MongoDbPostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor mongoDbPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<MongoOperations>() {

				@Override
				public Class<MongoOperations> getBeanType() {
					return MongoOperations.class;
				}

				@Override
				public void postProcessAfterInitialization(MongoOperations mongoOperations, String beanName) {
					//调用MongoDb初始化，由于第一次MongoDb调用会比较慢（尤其是其中Cat监控会给MongoDb加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量MongoDb调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					try {
						mongoOperations.findOne(Query.query(Criteria.where("id").is("1")), String.class,"test_collection");
					}catch(Exception e){
						logger.error(e.getMessage(), e);
					}
				}
			};
		}
	}

	/**
	 * 写点注释,迁移到dubbo里（这个方法废弃）
	 */
//	@Configuration
//	@ConditionalOnClass({ReferenceBean.class})
//	public static class DubboPostFactoryProcessorConfiguration{
//
//		@Bean
//		public static BeanFactoryPostProcessor dubboBeanFactoryPostProcessorConfigurer(){
//			return new DubboBeanFactoryPostProcessor();
//		}
//	}
//
//	private static class DubboBeanFactoryPostProcessor implements Ordered, BeanFactoryPostProcessor {
//		@Override
//		public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
//			for(String beanName : beanFactory.getBeanDefinitionNames()){
//				BeanDefinition bd = beanFactory.getBeanDefinition(beanName);
//				if(bd.isSingleton() && !bd.isAbstract() && !bd.isLazyInit() && ReferenceBean.class.getName().equals(bd.getBeanClassName())){
//					try {
//						beanFactory.getBean(beanName);
//					}catch(NoSuchBeanDefinitionException e){
//						// Ignore
//					}
//				}
//			}
//		}
//
//		@Override
//		public int getOrder() {
//			return Ordered.LOWEST_PRECEDENCE;
//		}
//	}
}
