package cn.com.duiba.boot.ext.autoconfigure.web.mvc;

import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.stereotype.Controller;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

import javax.servlet.http.HttpServletRequest;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 自定义的CustomRequestMappingHandlerMapping, 主要作用是方便重用FeignClient。
 * 目前官方spring mvc支持在父类方法中定义RequestMapping，子类可以不定义RequestMapping，然而如果此时父类参数加了RequestParam等注解，是不会生效的，必须写在子类中。这样的话父类和子类都必须写，非常繁琐。
 * 这个CustomRequestMappingHandlerMapping的作用就是让spring到声明RequestMapping的方法上去找参数注解
 *
 * Created by wenqi.huang on 2017/7/17.
 */
public class CustomRequestMappingHandlerMapping extends RequestMappingHandlerMapping {

    @Override
    protected boolean isHandler(Class<?> beanType) {
        return (AnnotatedElementUtils.hasAnnotation(beanType, Controller.class) ||
                AnnotatedElementUtils.hasAnnotation(beanType, RequestMapping.class)) && !beanType.isInterface();
    }

    @Override
    protected HandlerMethod getHandlerInternal(HttpServletRequest request) throws Exception {
        HandlerMethod m = super.getHandlerInternal(request);
        if(m == null){
            return m;
        }
        MethodParameter[] parameters = m.getMethodParameters();
        if(parameters == null){
            return m;
        }

        injectParameterAnnotations(parameters);

        return m;
//        return new CustomHandlerMethod(m);
    }

    //TODO 已经测试过与spring-boot 1.4.5/1.4.7版本(对应spring4.3.9）的兼容性，如果换用其他版本，这里可能会有问题，需要跟进
    /**
     * 这个方法的主要作用：判断如果RequestMapping没有注解在子类方法上，则到父类或父接口中寻找注解了RequestMapping的方法，并使用这个方法参数中的RequestParam/RequestBody等注解
     * @param parameters
     * @throws IllegalAccessException
     * @throws InvocationTargetException
     */
    private void injectParameterAnnotations(MethodParameter[] parameters) throws IllegalAccessException, InvocationTargetException {

        //这里获得的MethodParameter实际类型是HandlerMethod.HandlerMethodParameter
        for(MethodParameter p : parameters){
            Method method = p.getMethod();
            if(method == null){
                continue;
            }
            Field field = ReflectionUtils.findField(p.getClass(), "parameterAnnotations");//p.getParameterAnnotations()
            field.setAccessible(true);
            if(field.get(p) == null){
                Method requestMappingAnnotatedMethod = null;
                if(!isRequestMappingAnnotated(method)) {//优先到当前类或者父类中注解了@RequestMapping的方法中找对应参数的注解
                    Class[] interfaces = method.getDeclaringClass().getInterfaces();
                    Class superClass = method.getDeclaringClass().getSuperclass();
                    List<Class> classesToSearch = new ArrayList<>();
                    if(superClass != null || superClass != Object.class) {
                        classesToSearch.add(superClass);
                    }
                    classesToSearch.addAll(Arrays.asList(interfaces));
                    for (Class clazz : classesToSearch) {
                        try {
                            Method superMethod = clazz.getMethod(method.getName(), method.getParameterTypes());
                            if (isRequestMappingAnnotated(superMethod)){
                                requestMappingAnnotatedMethod = superMethod;
                                break;
                            }
                        } catch (NoSuchMethodException e) {
                            //Ignore
                        }
                    }
                }
                if(requestMappingAnnotatedMethod != null){
                    //优先从父类加载@RequestParam等注解的数据
                    Annotation[] parameterAnnotations = null;
                    Annotation[][] annotationArray = requestMappingAnnotatedMethod.getParameterAnnotations();
                    if (p.getParameterIndex() >= 0 && p.getParameterIndex() < annotationArray.length) {
                        //HandlerMethod.HandlerMethodParameter类中调用了adaptAnnotationArray方法，所以这里必须调用
                        //因为adaptAnnotationArray方法是protected，所以只能反射调用
//                        parameterAnnotations = p.adaptAnnotationArray(annotationArray[p.getParameterIndex()]);

                        Method adaptAnnotationArrayMethod = ReflectionUtils.findMethod(p.getClass(), "adaptAnnotationArray", Annotation[].class);
                        adaptAnnotationArrayMethod.setAccessible(true);
                        Object[] args = new Object[1];
                        Annotation[] anno = annotationArray[p.getParameterIndex()];
                        args[0] = anno;
                        parameterAnnotations = (Annotation[])adaptAnnotationArrayMethod.invoke(p, args);
                    }
                    else {
                        parameterAnnotations = new Annotation[0];
                    }
                    field.set(p, parameterAnnotations);

                    //确保这个方法存在
                    p.getParameterAnnotations();
                }
            }

        }
    }

    /**
     * 判断方法上是否有RequestMapping(GetMapping/PostMapping)注解
     */
    private boolean isRequestMappingAnnotated(Method m){
        Annotation[] arr = m.getAnnotations();
        for(Annotation a : arr){
            if(a instanceof RequestMapping || a.getClass().getAnnotation(RequestMapping.class)!=null){//后面这个表示GetMapping/PostMapping等注解
                return true;
            }
        }
        return false;
    }
}
