package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import io.prometheus.client.Collector;
import io.prometheus.client.CounterMetricFamily;
import io.prometheus.client.GaugeMetricFamily;
import io.prometheus.client.SummaryMetricFamily;
import net.bytebuddy.agent.ByteBuddyAgent;

import java.lang.instrument.Instrumentation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Collect metrics from Guava's com.google.common.cache.Cache.
 * <p>
 * <pre>{@code
 *
 * // Note that `recordStats()` is required to gather non-zero statistics
 * Cache<String, String> cache = CacheBuilder.newBuilder().recordStats().build();
 * CacheMetricsCollector cacheMetrics = new CacheMetricsCollector().register();
 * cacheMetrics.addCache("mycache", cache);
 *
 * }</pre>
 * <p>
 * Exposed metrics are labeled with the provided cache name.
 * <p>
 * With the example above, sample metric names would be:
 * <pre>
 *     guava_cache_hit_total{cache="mycache"} 10.0
 *     guava_cache_miss_total{cache="mycache"} 3.0
 *     guava_cache_requests_total{cache="mycache"} 13.0
 *     guava_cache_eviction_total{cache="mycache"} 1.0
 *     guava_cache_size{cache="mycache"} 5.0
 * </pre>
 * <p>
 * Additionally if the cache includes a loader, the following metrics would be provided:
 * <pre>
 *     guava_cache_load_failure_total{cache="mycache"} 2.0
 *     guava_cache_loads_total{cache="mycache"} 7.0
 *     guava_cache_load_duration_seconds_count{cache="mycache"} 7.0
 *     guava_cache_load_duration_seconds_sum{cache="mycache"} 0.0034
 * </pre>
 */
public class CacheMetricsCollector extends Collector {

    protected final ConcurrentMap<String, Cache> children = new ConcurrentHashMap<>();
    protected final ConcurrentMap<String, List<String>> labelsMap = new ConcurrentHashMap<>();

    /**
     * Add or replace the cache with the given name.
     * <p>
     * Any references any previous cache with this name is invalidated.
     *
     * @param cacheName The name of the cache, will be the metrics label value
     * @param cache     The cache being monitored
     */
    public void addCache(String cacheName, Cache cache, String instance, String service) {
        children.put(cacheName, cache);
        labelsMap.put(cacheName, Arrays.asList(instance, service));
    }

    /**
     * Remove the cache with the given name.
     * <p>
     * Any references to the cache are invalidated.
     *
     * @param cacheName cache to be removed
     */
    public Cache removeCache(String cacheName) {
        return children.remove(cacheName);
    }

    /**
     * Remove all caches.
     * <p>
     * Any references to all caches are invalidated.
     */
    public void clear() {
        children.clear();
    }

    public long getSize() {
        return children.size();
    }

    @Override
    public List<MetricFamilySamples> collect() {

        Instrumentation instrumentation = ByteBuddyAgent.install();

        List<MetricFamilySamples> mfs = new ArrayList<>();
        List<String> labelNames = Lists.newArrayList("cache");
        labelNames.add("instance");
        labelNames.add("service");

        CounterMetricFamily cacheHitTotal = new CounterMetricFamily("guava_cache_hit_total",
                "Cache hit totals", labelNames);
        mfs.add(cacheHitTotal);

        CounterMetricFamily cacheMissTotal = new CounterMetricFamily("guava_cache_miss_total",
                "Cache miss totals", labelNames);
        mfs.add(cacheMissTotal);

        CounterMetricFamily cacheRequestsTotal = new CounterMetricFamily("guava_cache_requests_total",
                "Cache request totals, hits + misses", labelNames);
        mfs.add(cacheRequestsTotal);

        CounterMetricFamily cacheEvictionTotal = new CounterMetricFamily("guava_cache_eviction_total",
                "Cache eviction totals, doesn't include manually removed entries", labelNames);
        mfs.add(cacheEvictionTotal);

        CounterMetricFamily cacheLoadFailure = new CounterMetricFamily("guava_cache_load_failure_total",
                "Cache load failures", labelNames);
        mfs.add(cacheLoadFailure);

        CounterMetricFamily cacheLoadTotal = new CounterMetricFamily("guava_cache_loads_total",
                "Cache loads: both success and failures", labelNames);
        mfs.add(cacheLoadTotal);

        GaugeMetricFamily cacheSize = new GaugeMetricFamily("guava_cache_size",
                "Cache size", labelNames);
        mfs.add(cacheSize);

        GaugeMetricFamily cacheMemory = new GaugeMetricFamily("guava_cache_memory",
                "Cache Memory", labelNames);
        mfs.add(cacheMemory);

        SummaryMetricFamily cacheLoadSummary = new SummaryMetricFamily("guava_cache_load_duration_seconds",
                "Cache load duration: both success and failures", labelNames);
        mfs.add(cacheLoadSummary);

        for (Map.Entry<String, Cache> c : children.entrySet()) {
            List<String> labelValues = Lists.newArrayList(c.getKey());
            labelValues.addAll(labelsMap.get(c.getKey()));
            CacheStats stats = c.getValue().stats();

            cacheHitTotal.addMetric(labelValues, stats.hitCount());
            cacheMissTotal.addMetric(labelValues, stats.missCount());
            cacheRequestsTotal.addMetric(labelValues, stats.requestCount());
            cacheEvictionTotal.addMetric(labelValues, stats.evictionCount());
            cacheSize.addMetric(labelValues, c.getValue().size());
            cacheMemory.addMetric(labelValues, instrumentation.getObjectSize(c.getValue()));

            cacheLoadFailure.addMetric(labelValues, stats.loadExceptionCount());
            cacheLoadTotal.addMetric(labelValues, stats.loadCount());
            cacheLoadSummary.addMetric(labelValues, stats.loadCount(), stats.totalLoadTime() / Collector.NANOSECONDS_PER_SECOND);
        }
        return mfs;
    }
}
