package cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Registration;
import com.esotericsoftware.kryo.factories.ReflectionSerializerFactory;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.CompatibleFieldSerializer;
import lombok.extern.slf4j.Slf4j;
import org.objenesis.strategy.StdInstantiatorStrategy;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;

/**
 * kryo序列化工具，仅供录制回放使用
 * Created by guoyanfei .
 * 2019-04-18 .
 */
@Slf4j
public class KryoSerializer {

    private static final int BUFFER_SIZE = 4096 * 4;

    private static final Class<?> ARRAY_AS_LIST;

    static {
        try {
            ARRAY_AS_LIST = Class.forName("java.util.Arrays$ArrayList");
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("should not be here", e);
        }
    }

    private static ThreadLocal<Kryo> kryoThreadLocal = ThreadLocal.withInitial(() -> {
        Kryo kryo = new Kryo(){
            public Registration writeClass (Output output, Class type) {
                if(type == ARRAY_AS_LIST){//java.util.Arrays$ArrayList用KRYO的默认实现在反序列化时会报错，这里用ArrayList代替
                    type = ArrayList.class;
                }
                return super.writeClass(output, type);
            }
        };

        //支持增删字段
        kryo.setDefaultSerializer(new ReflectionSerializerFactory(CompatibleFieldSerializer.class));
//		kryo.getFieldSerializerConfig().setFixedFieldTypes(true);
//		kryo.getFieldSerializerConfig().setOptimizedGenerics(true);
        kryo.setInstantiatorStrategy(new Kryo.DefaultInstantiatorStrategy(new StdInstantiatorStrategy()));
        return kryo;
    });

    private KryoSerializer() {
    }

    /**
     * 反序列化
     * @param bytes
     * @return
     */
    public static Object deserialize(byte[] bytes) {
        if (bytes == null || bytes.length == 0) {
            return null;
        }
        Kryo kryo = kryoThreadLocal.get();
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        Input input = new Input(in, BUFFER_SIZE);
        Object obj = kryo.readClassAndObject(input);
        input.close();
        return obj;
    }

    /**
     * 序列化
     * @param object
     * @return
     */
    public static byte[] serialize(Object object) {
        if (object == null) {
            return null;
        }
        Kryo kryo = kryoThreadLocal.get();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        Output output = new Output(out, BUFFER_SIZE);
        kryo.writeClassAndObject(output, object);
        output.close();
        return out.toByteArray();
    }

}
