package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import com.google.common.cache.Cache;
import com.google.common.collect.Maps;
import io.prometheus.client.exporter.PushGateway;
import lombok.extern.slf4j.Slf4j;
import org.ehcache.PersistentUserManagedCache;
import org.ehcache.config.builders.ResourcePoolsBuilder;
import org.ehcache.config.builders.UserManagedCacheBuilder;
import org.ehcache.config.units.EntryUnit;
import org.ehcache.config.units.MemoryUnit;
import org.ehcache.core.spi.service.LocalPersistenceService;
import org.ehcache.impl.config.persistence.DefaultPersistenceConfiguration;
import org.ehcache.impl.config.persistence.UserManagedPersistenceContext;
import org.ehcache.impl.persistence.DefaultLocalPersistenceService;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.annotation.Scheduled;

import java.io.File;
import java.lang.reflect.Field;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-07-17 15:55
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@EnableAsync
@ConditionalOnClass({Cache.class})
public class GuavaCacheMonitorConfiguration implements ApplicationContextAware {

    @Value("${spring.application.name}")
    private String currentAppName;

    @Value("${prometheus.pushgateway.address}")
    private String pushGatewayAddress;

    private final String path = System.getProperty("user.dir");

    private final ConcurrentMap<String, Cache> allCacheMap = Maps.newConcurrentMap();

    private final CacheMetricsCollector cacheMetrics = new CacheMetricsCollector().register();

    private final LocalPersistenceService persistenceService = new DefaultLocalPersistenceService(new DefaultPersistenceConfiguration(new File(path, "/monitor")));

    // 保存本地缓存的变化
    private final PersistentUserManagedCache<String, String> monitorCache = UserManagedCacheBuilder.newUserManagedCacheBuilder(String.class, String.class)
            .with(new UserManagedPersistenceContext<>("monitorCache", persistenceService))
            .withResourcePools(ResourcePoolsBuilder.newResourcePoolsBuilder()
                    .heap(10L, EntryUnit.ENTRIES)
                    .disk(10L, MemoryUnit.MB, true))
            .build(true);

    private ApplicationContext appContext;

    private final String localIp = NetUtils.getLocalIp();

    private final String prometheusJobName = "gauva_cache_monitor";

    private ScheduledExecutorService scheduledThreadPool = Executors.newScheduledThreadPool(1);

    @Async
    @EventListener(MainContextRefreshedEvent.class)
    public void loadAllCacheBean() {

        String[] beans = appContext.getBeanDefinitionNames();
        String curEnv = SpringEnvironmentUtils.getCurrentEnv();
        for (String beanName : beans) {

            Object beanObj = appContext.getBean(beanName);
            Field[] fields = beanObj.getClass().getDeclaredFields();
            if (fields.length == 0) {
                continue;
            }

            this.addCache(fields, beanName, beanObj, curEnv);
        }

        this.pushRecord();
    }

    /**
     * 保存所有缓存变量
     * @param fields
     * @param beanName
     * @param beanObj
     */
    private void addCache(Field[] fields, String beanName, Object beanObj, String curEnv) {

        for (Field field : fields) {

            if (!Cache.class.isAssignableFrom(field.getType())) {
                continue;
            }

            try {
                field.setAccessible(true);
                String keyName = String.format("%s-%s", beanName, field.getName());
                Cache cache = (Cache) field.get(beanObj);
                allCacheMap.put(keyName, cache);
                cacheMetrics.addCache(keyName, cache, localIp, currentAppName, curEnv);
            } catch (IllegalAccessException e) {
                //ignore
            } finally {
                field.setAccessible(false);
            }

        }
    }

    /**
     * 推送数据到pushGateway
     */
    private void pushRecord() {

        if(cacheMetrics.getSize() == 0) {
            return;
        }

        final PushGateway pushGateway = new PushGateway(pushGatewayAddress);
        scheduledThreadPool.scheduleAtFixedRate(() -> {

            try {
                pushGateway.pushAdd(cacheMetrics, prometheusJobName);
            } catch (Exception e) {
                log.error("推送指标到pushgateway异常", e);
            }

        }, 1, 1, TimeUnit.MINUTES);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.appContext = applicationContext;
    }
}
