/*
 * Copyright 2012-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.data.redis;

import cn.com.duiba.wolf.cache.Hessian2SerializationRedisSerializer;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.redis.connection.*;
import org.springframework.data.redis.connection.jedis.JedisConnection;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPoolConfig;

import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;

/**
 * {@link EnableAutoConfiguration Auto-configuration} for Spring Data's Redis support.
 * Redis自动配置
 * 注意：如果duiba.redis.host没有配置，则spring自带的RedisAutoConfiguration会生效
 *
 * @author hwq
 */
@Configuration
@AutoConfigureBefore(org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration.class)
@ConditionalOnClass({ JedisConnection.class, RedisOperations.class, Jedis.class })
@EnableConfigurationProperties(RedisProperties.class)
@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
public class RedisAutoConfiguration {

	//注释掉这几行代码以兼容springboot 1.4.5版本
//	@Bean(name = "org.springframework.autoconfigure.redis.RedisProperties")
//	@ConditionalOnMissingBean
//	@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
//	public RedisProperties redisProperties() {
//		return new RedisProperties();
//	}

	/**
	 * Base class for Redis configurations.
	 */
	protected abstract static class AbstractRedisConfiguration {

		@Autowired
		protected RedisProperties properties;

		@Autowired(required = false)
		private RedisSentinelConfiguration sentinelConfiguration;

		protected final JedisConnectionFactory applyProperties(
				JedisConnectionFactory factory) {
			factory.setHostName(this.properties.getHost());
			factory.setPort(this.properties.getPort());
			if (this.properties.getPassword() != null) {
				factory.setPassword(this.properties.getPassword());
			}
			factory.setDatabase(this.properties.getDatabase());
			if (this.properties.getTimeout() > 0) {
				factory.setTimeout(this.properties.getTimeout());
			}
			return factory;
		}

		protected final RedisSentinelConfiguration getSentinelConfig() {
			if (this.sentinelConfiguration != null) {
				return this.sentinelConfiguration;
			}
			RedisProperties.Sentinel sentinelProperties = this.properties.getSentinel();
			if (sentinelProperties != null) {
				RedisSentinelConfiguration config = new RedisSentinelConfiguration();
				config.master(sentinelProperties.getMaster());
				config.setSentinels(createSentinels(sentinelProperties));
				return config;
			}
			return null;
		}

		private List<RedisNode> createSentinels(RedisProperties.Sentinel sentinel) {
			List<RedisNode> sentinels = new ArrayList<>();
			String nodes = sentinel.getNodes();
			for (String node : StringUtils.commaDelimitedListToStringArray(nodes)) {
				try {
					String[] parts = StringUtils.split(node, ":");
					Assert.state(parts.length == 2, "Must be defined as 'host:port'");
					sentinels.add(new RedisNode(parts[0], Integer.valueOf(parts[1])));
				}
				catch (RuntimeException ex) {
					throw new IllegalStateException(
							"Invalid redis sentinel " + "property '" + node + "'", ex);
				}
			}
			return sentinels;
		}

	}

	/**
	 * Redis connection configuration.
	 */
	@Configuration
	@ConditionalOnClass({ JedisConnection.class, RedisOperations.class, Jedis.class })
	@ConditionalOnMissingClass("org.apache.commons.pool2.impl.GenericObjectPool")
	@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
	protected static class RedisConnectionConfiguration
			extends AbstractRedisConfiguration {

		@Bean(name="redisConnectionFactory")
		@ConditionalOnMissingBean(name="redisConnectionFactory")
		public JedisConnectionFactory redisConnectionFactory()
				throws UnknownHostException {
			return applyProperties(new JedisConnectionFactory(getSentinelConfig()));
		}

	}

	/**
	 * Redis pooled connection configuration.
	 */
	@Configuration
	@ConditionalOnClass({ JedisConnection.class, RedisOperations.class, Jedis.class, GenericObjectPool.class })
	@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
	protected static class RedisPooledConnectionConfiguration
			extends AbstractRedisConfiguration {

		@Bean(name="redisConnectionFactory")
		@ConditionalOnMissingBean(name="redisConnectionFactory")
		public JedisConnectionFactory redisConnectionFactory()
				throws UnknownHostException {
			return applyProperties(createJedisConnectionFactory());
		}

		private JedisConnectionFactory createJedisConnectionFactory() {
			if (this.properties.getPool() != null) {
				return new JedisConnectionFactory(getSentinelConfig(), jedisPoolConfig());
			}
			return new JedisConnectionFactory(getSentinelConfig());
		}

		private JedisPoolConfig jedisPoolConfig() {
			JedisPoolConfig config = new JedisPoolConfig();
			RedisProperties.Pool props = this.properties.getPool();
			config.setMaxTotal(props.getMaxActive());
			config.setMaxIdle(props.getMaxIdle());
			config.setMinIdle(props.getMinIdle());
			config.setMaxWaitMillis(props.getMaxWait());
			config.setTestWhileIdle(true);
			config.setTimeBetweenEvictionRunsMillis(90000);//每隔90S testWhileIdle一次

			// 按如下文章优化: https://mp.weixin.qq.com/s/6cJ5JuEgEWmMBzJFBDsSMg
			config.setSoftMinEvictableIdleTimeMillis(180000);
			config.setMinEvictableIdleTimeMillis(-1);
			config.setNumTestsPerEvictionRun(-1);

			return config;
		}

	}

	/**
	 * Standard Redis configuration.
	 */
	@Configuration
	@ConditionalOnClass({ JedisConnection.class, RedisOperations.class, Jedis.class })
	@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
	protected static class RedisConfiguration {

		@Bean(name="redisTemplate")
		@ConditionalOnMissingBean(name = "redisTemplate")
		public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory)
						throws UnknownHostException {
			RedisTemplate<Object, Object> template = new RedisTemplate<>();
			template.setConnectionFactory(redisConnectionFactory);
			//使用hession2序列化方式，提高性能，减少占用redis的空间
			template.setDefaultSerializer(new Hessian2SerializationRedisSerializer());
			return template;
		}

		@Bean(name = "stringRedisTemplate")
		@ConditionalOnMissingBean(name = "stringRedisTemplate")
		public StringRedisTemplate stringRedisTemplate(RedisConnectionFactory redisConnectionFactory)
						throws UnknownHostException {
			StringRedisTemplate template = new StringRedisTemplate();
			template.setConnectionFactory(redisConnectionFactory);
			return template;
		}

		@Bean
		public Hessian2RedisDeserializeMvcEndpoint hessian2RedisDeserializeMvcEndpoint() {
			return new Hessian2RedisDeserializeMvcEndpoint();
		}

	}

}
