package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Method;

/**
 * 引流回归，用于标记忽略子调用的线程变量
 */
public final class IgnoreSubInvokesContext {

    private final Object inst;
    private final Class clazz;
    private final Method method;
    private final Object[] allArguments;

    /**
     * CustomizeCallbackFlowReplaySpan 的 key
     */
    private final String key;

    private IgnoreSubInvokesContext(Object inst, Class clazz, Method method, Object[] allArguments) {
        this.inst = inst;
        this.clazz = clazz;
        this.method = method;
        this.allArguments = allArguments;
        this.key = null;
    }

    private IgnoreSubInvokesContext(String key) {
        this.inst = null;
        this.clazz = null;
        this.method = null;
        this.allArguments = null;
        this.key = key;
    }

    private static class IgnoreSubInvokesThreadLocalHolder {

        private IgnoreSubInvokesThreadLocalHolder() {
        }

        /**
         * 忽略子调用的标记，比如：缓存load方法中的一些子调用忽略掉
         */
        private static final ThreadLocal<IgnoreSubInvokesContext> ignoreSubInvokesThreadLocal = new ThreadLocal<>();

    }

    /**
     * 标记key
     * @param key
     */
    public static void keyMark(String key) {
        if (StringUtils.isBlank(key)) {
            return;
        }
        IgnoreSubInvokesContext c = new IgnoreSubInvokesContext(key);
        IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.set(c);
    }

    /**
     * 标记实例
     * @param inst
     * @param method
     * @param allArguments
     */
    public static void instMark(Object inst, Method method, Object[] allArguments) {
        if (inst == null || method == null) {
            return;
        }
        IgnoreSubInvokesContext c = new IgnoreSubInvokesContext(inst, null, method, allArguments);
        IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.set(c);
    }

    /**
     * 标记静态方法
     * @param clazz
     * @param method
     * @param allArguments
     */
    public static void staticMark(Class clazz, Method method, Object[] allArguments) {
        if (clazz == null || method == null) {
            return;
        }
        IgnoreSubInvokesContext c = new IgnoreSubInvokesContext(null, clazz, method, allArguments);
        IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.set(c);
    }

    /**
     * 已经被标记
     * @return
     */
    public static boolean isMarked() {
        IgnoreSubInvokesContext c = IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.get();
        return c != null;
    }

    /**
     * 当前key是否和被标记的key相同
     * @param key
     * @return
     */
    public static boolean isKeyMarked(String key) {
        if (StringUtils.isBlank(key)) {
            return false;
        }
        IgnoreSubInvokesContext c = IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.get();
        if (c == null) {
            return false;
        }
        return key.equals(c.key);
    }

    /**
     * 当前实例是否是被标记的实例
     * @param inst
     * @param method
     * @param allArguments
     * @return
     */
    public static boolean isInstMarked(Object inst, Method method, Object[] allArguments) {
        if (inst == null || method == null) {
            return false;
        }
        IgnoreSubInvokesContext c = IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.get();
        if (c == null) {
            return false;
        }
        return c.inst == inst && c.method == method && c.allArguments == allArguments;
    }

    /**
     * 当前静态方法是否是被标记的静态方法
     * @param clazz
     * @param method
     * @param allArguments
     * @return
     */
    public static boolean isStaticMarked(Class clazz, Method method, Object[] allArguments) {
        if (clazz == null || method == null) {
            return false;
        }
        IgnoreSubInvokesContext c = IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.get();
        if (c == null) {
            return false;
        }
        return c.clazz == clazz && c.method == method && c.allArguments == allArguments;
    }

    /**
     * 取消标记
     */
    public static void unmark() {
        IgnoreSubInvokesThreadLocalHolder.ignoreSubInvokesThreadLocal.remove();
    }
}
