package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import com.google.common.cache.CacheStats;
import com.google.common.collect.Lists;
import io.prometheus.client.Collector;
import io.prometheus.client.CounterMetricFamily;
import io.prometheus.client.GaugeMetricFamily;
import net.bytebuddy.agent.ByteBuddyAgent;

import java.lang.instrument.Instrumentation;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * Collect metrics from Guava's com.google.common.cache.Cache.
 * <p>
 * <pre>{@code
 *
 * // Note that `recordStats()` is required to gather non-zero statistics
 * Cache<String, String> cache = CacheBuilder.newBuilder().recordStats().build();
 * CacheMetricsCollector cacheMetrics = new CacheMetricsCollector().register();
 * cacheMetrics.addCache("mycache", cache);
 *
 * }</pre>
 * <p>
 * Exposed metrics are labeled with the provided cache name.
 * <p>
 * With the example above, sample metric names would be:
 * <pre>
 *     guava_cache_hit_total{cache="mycache"} 10.0
 *     guava_cache_miss_total{cache="mycache"} 3.0
 *     guava_cache_requests_total{cache="mycache"} 13.0
 *     guava_cache_eviction_total{cache="mycache"} 1.0
 *     guava_cache_size{cache="mycache"} 5.0
 * </pre>
 * <p>
 * Additionally if the cache includes a loader, the following metrics would be provided:
 * <pre>
 *     guava_cache_load_failure_total{cache="mycache"} 2.0
 *     guava_cache_loads_total{cache="mycache"} 7.0
 *     guava_cache_load_duration_seconds_count{cache="mycache"} 7.0
 *     guava_cache_load_duration_seconds_sum{cache="mycache"} 0.0034
 * </pre>
 */
public class CacheMetricsCollector extends Collector {

    protected final ConcurrentMap<String, Object> children = new ConcurrentHashMap<>();
    protected final ConcurrentMap<String, List<String>> labelsMap = new ConcurrentHashMap<>();

    /**
     * Add or replace the cache with the given name.
     * <p>
     * Any references any previous cache with this name is invalidated.
     *
     * @param cacheName The name of the cache, will be the metrics label value
     * @param cache     The cache being monitored
     */
    public void addCache(String cacheName, Object cache, String ip, String service, String namespace) {
        children.put(cacheName, cache);
        labelsMap.put(cacheName, Arrays.asList(ip, service, namespace));
    }

    /**
     * Remove the cache with the given name.
     * <p>
     * Any references to the cache are invalidated.
     *
     * @param cacheName cache to be removed
     */
    public void removeCache(String cacheName) {
        children.remove(cacheName);
    }

    /**
     * Remove all caches.
     * <p>
     * Any references to all caches are invalidated.
     */
    public void clear() {
        children.clear();
    }

    public long getSize() {
        return children.size();
    }

    @Override
    public List<MetricFamilySamples> collect() {

        Instrumentation instrumentation = ByteBuddyAgent.install();

        List<MetricFamilySamples> mfs = new ArrayList<>();
        List<String> labelNames = Lists.newArrayList("cache");
        labelNames.add("ip");
        labelNames.add("service");
        labelNames.add("namespace");

        CounterMetricFamily cacheHitRateTotal = new CounterMetricFamily("guava_cache_hit_rate",
                "缓存命中率", labelNames);
        mfs.add(cacheHitRateTotal);

        CounterMetricFamily cacheFailRateTotal = new CounterMetricFamily("guava_cache_fail_rate",
                "加载错误率", labelNames);
        mfs.add(cacheFailRateTotal);

        CounterMetricFamily cacheRequestsTotal = new CounterMetricFamily("guava_cache_requests_total",
                "缓存请求数：命中 + 未命中 ", labelNames);
        mfs.add(cacheRequestsTotal);

        CounterMetricFamily cacheEvictionTotal = new CounterMetricFamily("guava_cache_eviction_total",
                "缓存回收总数，不包括显示清除的", labelNames);
        mfs.add(cacheEvictionTotal);

        GaugeMetricFamily cacheSize = new GaugeMetricFamily("guava_cache_size",
                "缓存总数", labelNames);
        mfs.add(cacheSize);

        GaugeMetricFamily cacheMemory = new GaugeMetricFamily("guava_cache_memory",
                "缓存占用内存", labelNames);
        mfs.add(cacheMemory);

        GaugeMetricFamily cacheLoadSummary = new GaugeMetricFamily("guava_cache_load_duration_seconds",
                "缓存平均加载时间，包括加载失败和成功", labelNames);
        mfs.add(cacheLoadSummary);

        for (Map.Entry<String, Object> c : children.entrySet()) {

            List<String> labelValues = Lists.newArrayList(c.getKey());
            labelValues.addAll(labelsMap.get(c.getKey()));

            if (c.getValue() instanceof com.google.common.cache.Cache) {

                com.google.common.cache.Cache cache = (com.google.common.cache.Cache) c.getValue();
                CacheStats stats = cache.stats();

                cacheHitRateTotal.addMetric(labelValues, stats.hitRate());
                cacheFailRateTotal.addMetric(labelValues, stats.loadExceptionRate());
                cacheRequestsTotal.addMetric(labelValues, stats.requestCount());
                cacheLoadSummary.addMetric(labelValues, stats.averageLoadPenalty());

                cacheEvictionTotal.addMetric(labelValues, stats.evictionCount());
                cacheSize.addMetric(labelValues, cache.size());
                cacheMemory.addMetric(labelValues, instrumentation.getObjectSize(cache));
            }

            if(c.getValue() instanceof com.github.benmanes.caffeine.cache.Cache) {

                com.github.benmanes.caffeine.cache.Cache cache = (com.github.benmanes.caffeine.cache.Cache) c.getValue();
                com.github.benmanes.caffeine.cache.stats.CacheStats stats = cache.stats();

                cacheHitRateTotal.addMetric(labelValues, stats.hitRate());
                cacheFailRateTotal.addMetric(labelValues, stats.loadFailureRate());
                cacheRequestsTotal.addMetric(labelValues, stats.requestCount());
                cacheLoadSummary.addMetric(labelValues, stats.averageLoadPenalty());

                cacheEvictionTotal.addMetric(labelValues, stats.evictionCount());
                cacheSize.addMetric(labelValues, cache.estimatedSize());
                cacheMemory.addMetric(labelValues, instrumentation.getObjectSize(cache));
            }

        }
        return mfs;
    }
}
