package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.RocketMqProducerFlowReplaySpan;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.Order;

import java.lang.reflect.Method;

/**
 * 录制RocketMq的方法
 * Created by guoyanfei .
 * 2019-07-12 .
 */
@Slf4j
@Aspect
@Order(1)
public class RecordRocketMqProducerPlugin {

    @Around("execution(* org.apache.rocketmq.client.producer.DefaultMQProducer+.*(..))")
    public Object rocketMqProducerJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();

        if (!FlowReplayTrace.isTraced() || !"send".equals(method.getName())) {
            return joinPoint.proceed();
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return joinPoint.proceed();
        }

        Object[] parameterValues = joinPoint.getArgs();
        Object obj = joinPoint.getTarget();
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method, parameterValues);

        RocketMqProducerFlowReplaySpan span = null;
        try {
            span = RocketMqProducerFlowReplaySpan.createSpan(signature, parameterValues);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
            log.debug("RocketMq录制_traceId={}_spanId={}", span.getTraceId(), span.getSpanId());
        } catch (Throwable t) {
            log.error("RocketMq录制_录制异常", t);
        } finally {
            IgnoreSubInvokesContext.unmark();
        }

        Object ret;
        try {
            ret = joinPoint.proceed();
        } catch (Throwable t) {
            // 如果捕捉到异常，并且正在录制中，那么不录这个用例
            FlowReplayTrace.remove();
            IgnoreSubInvokesContext.unmark();
            throw t;
        }

        try {
            if (span != null) {
                span.setReturnValue(ret);
            }
        } catch (Throwable t) {
            log.error("RocketMq录制_录制异常", t);
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        
        return ret;
    }
}
