package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayEndEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.event.ReplayStartEvent;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.replayer.Replayer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * 多线程消费用例加载器加载到内存中的用例，调用回放器回放对应类型的用例
 * Created by guoyanfei .
 * 2019-02-26 .
 */
@Slf4j
public class ReplayTraceReplayer {

    /**
     * 回放器
     */
    @Resource
    private Replayer replayer;

    private volatile ExecutorService replayerThreadPool;

    private volatile Thread replayThread;

    /**
     * 监听"回放开始"事件，多线程启动多个回放器
     * @param event
     * @throws ExecutionException
     * @throws InterruptedException
     */
    @EventListener(ReplayStartEvent.class)
    public void replayStartEventListener(ReplayStartEvent event) {
        replayThread = new Thread(new Runnable() {
            @Override
            public void run() {
                ReplayContext context = ReplayContextHolder.getReplayContext();
                if (context == null) {
                    return;
                }
                int replayThreadPoolSize = context.getReplayThreadPoolSize();
                replayerThreadPool = Executors.newFixedThreadPool(replayThreadPoolSize, new NamedThreadFactory("replay", true));
                try {
                    List<Future> futures = new ArrayList<>(replayThreadPoolSize);
                    for (int i = 0; i < replayThreadPoolSize; i++) {
                        try {
                            Thread.sleep(2000);
                        } catch (InterruptedException e) {
                            Thread.currentThread().interrupt();
                        }
                        if (Thread.currentThread().isInterrupted()) {
                            break;
                        }
                        futures.add(replayerThreadPool.submit(new ReplayerThread(context)));
                    }
                    for (Future it : futures) {
                        try {
                            log.debug("引流回归_开始等待回放结果_reportId_{}", context.getReportId());
                            it.get();
                        } catch (Exception e) {
                            log.error("引流回归_ReplayerThread_Future_get_error", e);
                        }
                    }
                } finally {
                    // 标记回放操作已经完成，但是结果队列可能还在写文件
                    context.replayersDone();
                }
            }
        }, "DBThread-Replay-Replayer");

        replayThread.start();
    }

    @EventListener(ReplayEndEvent.class)
    public void replayEndEventListener(ReplayEndEvent event) {
        if (replayerThreadPool != null) {
            // 当线程池提交的任务阻塞的时候 调用 replayerThreadPool.shutdown() 无法关闭线程池，需要调用 replayerThreadPool.shutdownNow()
            replayerThreadPool.shutdownNow();
        }
        if (replayThread != null) {
            replayThread.interrupt();
        }
    }

    /**
     * 用例消费者线程
     */
    private class ReplayerThread implements Runnable {

        private ReplayContext context;

        public ReplayerThread(ReplayContext context) {
            this.context = context;
        }

        @Override
        public void run() {
            try {
                replay();
            } catch (Exception e) {
                log.error("引流回归_回放器线程异常_reportId_{}", context.getReportId(), e);
                ReplayContextHolder.forceEnd();
            }
        }

        private void replay() {
            // (用例集未加载完成 || trace队列不是空的) 消费队列中的trace
            while (!context.isTracesLoadFinished() || !context.isTraceQueueEmpty()) {
                try {
                    FlowReplayTrace trace = context.pollTrace();
                    if (trace == null) {
                        continue;
                    }
                    ReplayTraceResult traceResult = replayer.replay(trace);
                    context.putResult(traceResult);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                } catch (Exception e) {
                    log.error("引流回归_单个用例回放异常_reportId_{}", context.getReportId(), e);
                }
                if (Thread.currentThread().isInterrupted()) {
                    break;
                }
            }
        }
    }
}
