package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duibaboot.ext.autoconfigure.etcd.EtcdAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.EtcdKVClientDelegate;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import io.prometheus.client.exporter.PushGateway;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.scheduling.annotation.Async;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import javax.servlet.Servlet;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-07-17 15:55
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@EnableAsync
@ConditionalOnClass({CacheBuilder.class, Caffeine.class})
@AutoConfigureAfter(EtcdAutoConfiguration.class)
public class CacheMonitorConfiguration implements ApplicationContextAware, InitializingBean {

    @Value("${spring.application.name}")
    private String currentAppName;

    @Value("${prometheus.pushgateway.address}")
    private String pushGatewayAddress;

    private final CacheMetricsCollector cacheMetrics = new CacheMetricsCollector().register();

    private ApplicationContext appContext;

    private final String localIp = NetUtils.getLocalIp();

    private final String prometheusJobName = "cache_monitor";

    private ScheduledExecutorService scheduledThreadPool = Executors.newScheduledThreadPool(1);

    @Resource
    private EtcdKVClientDelegate etcdKVClientDelegate;

    @Async
    @EventListener(MainContextRefreshedEvent.class)
    public void loadAllCacheBean() {

        String[] beans = appContext.getBeanDefinitionNames();
        String curEnv = SpringEnvironmentUtils.getCurrentEnv();
        for (String beanName : beans) {

            Object beanObj = appContext.getBean(beanName);
            Field[] fields = beanObj.getClass().getDeclaredFields();
            if (fields.length == 0) {
                continue;
            }

            this.addCache(fields, beanName, beanObj, curEnv);
        }

        this.pushRecord();
    }

    /**
     * 保存所有缓存变量
     *
     * @param fields
     * @param beanName
     * @param beanObj
     */
    private void addCache(Field[] fields, String beanName, Object beanObj, String curEnv) {

        for (Field field : fields) {

            boolean isGuava = Cache.class.isAssignableFrom(field.getType());
            boolean isCaffeine = com.github.benmanes.caffeine.cache.Cache.class.isAssignableFrom(field.getType());

            if (!isGuava && !isCaffeine) {
                continue;
            }

            field.setAccessible(true);
            String keyName = String.format("%s-%s", beanName, field.getName());

            try {

                Object cache = null;
                if (isGuava) {
                    cache = field.get(beanObj);
                    CacheMonitorManager.saveCache(cache, keyName);
                }

                if (isCaffeine) {
                    cache = field.get(beanObj);
                    CacheMonitorManager.saveCache(cache, keyName);
                }

                cacheMetrics.addCache(keyName, cache, localIp, currentAppName, curEnv);
            } catch (IllegalAccessException e) {
                //ignore
            } finally {
                field.setAccessible(false);
            }

        }
    }

    /**
     * 推送数据到pushGateway
     */
    private void pushRecord() {

        if (cacheMetrics.getSize() == 0) {
            return;
        }

        final PushGateway pushGateway = new PushGateway(pushGatewayAddress);
        scheduledThreadPool.scheduleAtFixedRate(() -> {

            try {
                pushGateway.pushAdd(cacheMetrics, prometheusJobName);
            } catch (Exception e) {
                log.error("推送指标到pushgateway异常", e);
            }

        }, 1, 1, TimeUnit.MINUTES);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.appContext = applicationContext;
    }

    @Override
    public void afterPropertiesSet() {
        CacheMonitorManager.init(etcdKVClientDelegate, currentAppName);
    }


    @Configuration
    @ConditionalOnClass({Servlet.class, DispatcherServlet.class, WebMvcConfigurerAdapter.class})
    @ConditionalOnWebApplication
    public static class CacheMonitorFilterConfiguration {
        @Bean
        public FilterRegistrationBean cacheMonitorFilterConfiguration() {
            CacheMonitorFilter filter = new CacheMonitorFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns = new ArrayList<>();
            urlPatterns.add("/monitor/cache/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 1);//最高优先级
            return registrationBean;
        }
    }
}
