package cn.com.duibaboot.ext.autoconfigure.monitor.cache;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.EtcdKVClientDelegate;
import com.alibaba.fastjson.JSON;
import com.google.common.cache.Cache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.ehcache.PersistentUserManagedCache;
import org.ehcache.config.builders.ResourcePoolsBuilder;
import org.ehcache.config.builders.UserManagedCacheBuilder;
import org.ehcache.config.units.EntryUnit;
import org.ehcache.config.units.MemoryUnit;
import org.ehcache.core.spi.service.LocalPersistenceService;
import org.ehcache.impl.config.persistence.DefaultPersistenceConfiguration;
import org.ehcache.impl.config.persistence.UserManagedPersistenceContext;
import org.ehcache.impl.persistence.DefaultLocalPersistenceService;

import java.io.File;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;


/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-07-24 15:44
 * @descript:
 * @version: 1.0
 */
@Slf4j
class CacheMonitorManager {

    private static EtcdKVClientDelegate etcdKVClientDelegate;

    private static final String localIp = NetUtils.getLocalIp();

    private static String currentAppName;

    private static final ConcurrentMap<Object, String> allCacheMap = Maps.newConcurrentMap();

    private static final ConcurrentMap<Integer, Object> cacheBuilderMap = Maps.newConcurrentMap();

    private static final Queue<CacheChangeRecord> recordQueue = new ConcurrentLinkedQueue<>();

    private static final LocalPersistenceService persistenceService = new DefaultLocalPersistenceService(new DefaultPersistenceConfiguration(new File(System.getProperty("user.home"), "/monitor")));

    // 保存本地缓存的变化
    private static final PersistentUserManagedCache<String, ArrayList> monitorCache = UserManagedCacheBuilder.newUserManagedCacheBuilder(String.class, ArrayList.class)
            .with(new UserManagedPersistenceContext<>("monitorCache", persistenceService)).withResourcePools(ResourcePoolsBuilder.newResourcePoolsBuilder()
                    .heap(1, MemoryUnit.MB)//堆内存最大占用
                    .disk(100L, MemoryUnit.MB, true))//磁盘占用量
            .build(true);

    private static ScheduledExecutorService scheduledThreadPool = Executors.newScheduledThreadPool(1);

    static {

        scheduledThreadPool.scheduleAtFixedRate(() -> {

            if (recordQueue.size() == 0) {
                return;
            }

            for (int i = 0, len = recordQueue.size(); i < len; i++) {

                CacheChangeRecord newRecord = recordQueue.poll();
                if (newRecord == null || StringUtils.isBlank(newRecord.getCacheName())) {
                    continue;
                }

                String key = String.format("%s-%s", newRecord.getCacheName(), String.valueOf(newRecord.getKey()));
                ArrayList oldRecords = monitorCache.get(key);
                if (CollectionUtils.isEmpty(oldRecords)) {
                    oldRecords = Lists.newArrayList();
                    newRecord.buildSerializableObj();
                    oldRecords.add(newRecord);
                    monitorCache.put(key, oldRecords);
                } else {
                    CacheChangeRecord lastRecord = (CacheChangeRecord) oldRecords.get(oldRecords.size() - 1);
                    if (!newRecord.equals(lastRecord)) {
                        newRecord.buildSerializableObj();
                        oldRecords.add(newRecord);
                        monitorCache.put(key, oldRecords);
                    }
                }

            }

        }, 10, 10, TimeUnit.SECONDS);

    }

    static void saveCache(Object key, String value) {
        allCacheMap.put(key, value);
    }

    static Set<String> getAllCache() {

        Set<String> names = Sets.newHashSet();
        for (Map.Entry<Object, String> entry : allCacheMap.entrySet()) {
            names.add(entry.getValue());
        }

        return names;
    }

    static Object getCacheValueByKey(String cacheName, String key) {

        Object cacheObj = null;
        for (Map.Entry<Object, String> entry : allCacheMap.entrySet()) {

            if (entry.getValue().equals(cacheName)) {
                cacheObj = entry.getKey();
                break;
            }
        }

        if (cacheObj == null) {
            return null;
        }

        if (cacheObj instanceof Cache) {

            Cache cache = (Cache) cacheObj;
            Iterator iterator = cache.asMap().keySet().iterator();
            if (!iterator.hasNext()) {
                return null;
            }

            Class aClass = iterator.next().getClass();
            return cache.getIfPresent(JSON.parseObject(key, aClass));
        }

        if (cacheObj instanceof com.github.benmanes.caffeine.cache.Cache) {
            com.github.benmanes.caffeine.cache.Cache cache = (com.github.benmanes.caffeine.cache.Cache) cacheObj;
            Iterator iterator = cache.asMap().keySet().iterator();
            if (!iterator.hasNext()) {
                return null;
            }

            Class aClass = iterator.next().getClass();
            return cache.getIfPresent(JSON.parseObject(key, aClass));
        }

        return null;
    }

    static RecordPagerResponse<CacheChangeRecord> getRecord(String cacheName, String key, Integer pageSize, Integer page) {

        if (StringUtils.isBlank(cacheName) || StringUtils.isBlank(key)) {
            return null;
        }

        if (page == null) {
            page = 1;
        }

        if (pageSize == null) {
            pageSize = 20;
        }

        String cacheKey = String.format("%s-%s", cacheName, key);
        List<CacheChangeRecord> values = monitorCache.get(cacheKey);
        if (CollectionUtils.isEmpty(values)) {
            return null;
        }

        Collections.reverse(values);
        List<CacheChangeRecord> pageValues = values.stream().skip((page - 1) * pageSize).limit(pageSize).collect(Collectors.toList());
        return new RecordPagerResponse(page, pageSize, values.size(), pageValues);
    }

    static void setBuilderCache(Integer hashcode, Object value) {
        cacheBuilderMap.put(hashcode, value);
    }

    static void addRecord(CacheChangeRecord record) {

        if (etcdKVClientDelegate == null) {
            log.warn("has no bean etcdKVClientDelegate");
            return;
        }

        if (StringUtils.isBlank(currentAppName)) {
            log.warn("has no currentAppName");
            return;
        }

        Object cache = cacheBuilderMap.get(record.getBuilderHashcode());
        if (cache == null) {
            return;
        }

        String cacheName = allCacheMap.get(cache);
        if (StringUtils.isBlank(cacheName)) {
            return;
        }

        String path = String.format(CacheMonitorConstant.RECORD_ETCD_PATH_FORMAT, currentAppName, localIp, cacheName);
        String conifg = etcdKVClientDelegate.get(path);
        if ("true".equals(conifg)) {
            record.setCacheName(cacheName);
            recordQueue.offer(record);
        }
    }

    static void init(EtcdKVClientDelegate bean, String appName) {

        if (etcdKVClientDelegate == null) {
            etcdKVClientDelegate = bean;
        }

        if (appName != null) {
            currentAppName = appName;
        }
    }
}


