package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka;

import com.netflix.appinfo.ApplicationInfoManager;
import com.netflix.appinfo.HealthCheckHandler;
import com.netflix.discovery.EurekaClient;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.cloud.client.discovery.event.InstanceRegisteredEvent;
import org.springframework.cloud.netflix.eureka.CloudEurekaInstanceConfig;
import org.springframework.cloud.netflix.eureka.serviceregistry.EurekaAutoServiceRegistration;
import org.springframework.cloud.netflix.eureka.serviceregistry.EurekaRegistration;
import org.springframework.cloud.netflix.eureka.serviceregistry.EurekaServiceRegistry;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;

import javax.annotation.PostConstruct;

/**
 * eurekaClient实例在 @ConditionalOnRefreshScope 情况下每次 refresh，会销毁重新创建。实例会取消注册，再重新注册到eureka。导致其他应用从eureka获取该应用实例的时候存在找不到的情况
 * 此处扩展后，refresh不再会重新创建eurekaClient实例
 * Created by gyf .
 * 2018/4/27 .
 */
@Configuration
@ConditionalOnClass(EurekaAutoServiceRegistration.class)
@Slf4j
public class EurekaClientPostProcessorAutoConfiguration {

    //BeanDefinitionRegistryPostProcessor的声明必须是static的
    @Bean
    public static EurekaClientBeanDefinitionRegistryPostProcessor eurekaClientBeanDefinitionRegistryPostProcessor() {
        return new EurekaClientBeanDefinitionRegistryPostProcessor();
    }

    //拷贝官方的实现
    @Bean
    public EurekaRegistration eurekaRegistration(EurekaClient eurekaClient,
                                                 CloudEurekaInstanceConfig instanceConfig,
                                                 ApplicationInfoManager applicationInfoManager,
                                                 @Autowired(required = false) HealthCheckHandler healthCheckHandler) {
        return EurekaRegistration.builder(instanceConfig)
                .with(applicationInfoManager)
                .with(eurekaClient)
                .with(healthCheckHandler)
                .build();
    }

    /**
     * eureka自动注册和取消注册的bean
     * @param context
     * @param registry
     * @param registration
     * @return
     */
    @Bean
    public static DuibaEurekaAutoServiceRegistration duibaEurekaAutoServiceRegistration(ApplicationContext context, EurekaServiceRegistry registry,
                                                                                        EurekaRegistration registration) {
        return new DuibaEurekaAutoServiceRegistration(context, registry, registration);
    }

    @EventListener(InstanceRegisteredEvent.class)
    public void onEureuaRegistered(InstanceRegisteredEvent e){
        log.info("当前实例已经成功注册到eureka");
    }

    @Autowired(required = false)
    private EurekaAutoServiceRegistration autoRegistration;
    @Autowired(required = false)
    private EurekaClient eurekaClient;

    @PostConstruct
    public void init(){
        if(autoRegistration != null){
            throw new IllegalStateException("请关闭spring-cloud官方的自动注册功能，默认情况下我们会添加默认配置：spring.cloud.service-registry.auto-registration.enabled=false来关闭该功能，请检查你是否自己设置了这个配置,如有，请去除");
        }
        if(eurekaClient != null){
            if(AopUtils.isAopProxy(eurekaClient)){
                throw new IllegalStateException("检测到eurekaClient实例是经过代理的，这应该是spring-cloud新版本变更过相关代码导致的，这可能会导致/refresh的时候本实例不再注册到eureka上，如遇到此错误，请联系中间件团队修复此问题");
            }
        }
    }

}
