package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duiba.wolf.perf.timeprofile.DBTimeProfile;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.SpringMvcFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordContextHolder;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler.RecordSampler;
import cn.com.duibaboot.ext.autoconfigure.web.wrapper.BodyReaderHttpServletRequestWrapper;
import cn.com.duibaboot.ext.autoconfigure.web.wrapper.BodyWriterHttpServletResponseWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.MediaType;
import org.springframework.web.servlet.HandlerMapping;

import javax.annotation.Resource;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

/**
 * 录制 SpringMvc Controller 的过滤器
 * Created by guoyanfei .
 * 2019-03-05 .
 */
@Slf4j
public class RecordSpringMvcFilter implements Filter {

    private static final long MAX_CONTENT_LENGTH = 10 * 1024 * 1024;

    @Resource
    private RecordSampler recordSampler;

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        // do nothing
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest req = (HttpServletRequest) request;
        HttpServletResponse resp = (HttpServletResponse) response;

        SpringMvcFlowReplaySpan mainSpan = null;
        BodyReaderHttpServletRequestWrapper requestWrapper = null;
        BodyWriterHttpServletResponseWrapper responseWrapper = null;
        if (canRecord(req)) {
            requestWrapper = new BodyReaderHttpServletRequestWrapper(req);
            responseWrapper = new BodyWriterHttpServletResponseWrapper(resp);

            try {
                FlowReplayTrace.cleanRecordEnv();

                mainSpan = SpringMvcFlowReplaySpan.createSpan(requestWrapper);
                FlowReplayTrace.createTrace(mainSpan);
                mainSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
            } catch (Exception e) {
                log.error("SpringMvcFlowReplaySpan_createSpan_error", e);
            }

            IgnoreSubInvokesContext.unmark();
        }

        try {
            chain.doFilter(requestWrapper == null ? req : requestWrapper, responseWrapper == null ? resp : responseWrapper);
        } finally {
            this.tryStoreTrace(mainSpan, requestWrapper, responseWrapper);
        }
    }

    /**
     * 能否录制本次请求
     * @param request
     * @return
     */
    private boolean canRecord(HttpServletRequest request) {
        // 当前应用没有开启录制，不录制
        if (!RecordContextHolder.isRecording()) {
            return false;
        }
        // 本次请求是feign请求，不录制
        if (isFeignRequest(request)) {
            return false;
        }
        // 本次请求是排除列表的，不录制
        if (FlowReplayConstants.HTTP_EXCLUDE_URI.contains(request.getRequestURI())) {
            return false;
        }
        // 本次请求的 content 不符合要求的，不录制
        if (!canRequestContentRecord(request)) {
            return false;
        }
        return recordSampler.isSampled();
    }

    private void tryStoreTrace(SpringMvcFlowReplaySpan mainSpan, BodyReaderHttpServletRequestWrapper requestWrapper, BodyWriterHttpServletResponseWrapper responseWrapper) {
        // 正常请求，非录制请求
        if (requestWrapper == null || responseWrapper == null) {
            return;
        }

        try {
            DBTimeProfile.setCurrentThreshold(FlowReplayConstants.DB_TIME_PROFILE_THRESHOLD);
            byte[] responseBody = responseWrapper.getResponseBody();
            // 响应的body类型可以录制
            if (canResponseContentRecord(responseWrapper, responseBody)) {
                RecordContext context = RecordContextHolder.getRecordContext();
                if (context == null || mainSpan == null) {
                    return;
                }
                mainSpan.setResponseBody(responseBody);
                mainSpan.setSpringBestMatchingUri((String) requestWrapper.getAttribute(HandlerMapping.BEST_MATCHING_PATTERN_ATTRIBUTE));
                FlowReplayTrace trace = FlowReplayTrace.get();
                if (trace != null) {
                    context.offerTrace(trace);
                }
            }
        } catch (Throwable t) {
            log.error("tryStoreTrace_error", t);
        } finally {
            FlowReplayTrace.remove();
        }
    }

    /**
     * 判断响应的内容，是不是可以录制
     * 控制器中 @ResponseBody 注解的方法
     * 如果返回值类型是一个声明好的类，那么 Content-Type="application/json;charset=UTF-8"，此时只需要判断Content-Type即可
     * 如果返回值类型是一个String，那么 Content-Type="text/plain;charset=UTF-8"，此时还需要判断 responseBody 是否是一个json
     * @param response
     * @param responseBody
     * @return
     */
    private boolean canResponseContentRecord(HttpServletResponse response, byte[] responseBody) {
        String contentType = response.getContentType();
        if (contentType == null || contentType.contains(MediaType.TEXT_PLAIN_VALUE)) {
            return FlowReplayUtils.isJSONValid(new String(responseBody, StandardCharsets.UTF_8));
        }
        return contentType.contains(MediaType.APPLICATION_JSON_VALUE);
    }

    /**
     * 判断请求的内容，是不是可以录制
     * @param request
     * @return
     */
    private boolean canRequestContentRecord(HttpServletRequest request) {
        String contentType = request.getHeader("Content-Type");
        if (contentType == null) {
            return true;
        }
        String contentLengthStr = request.getHeader("Content-Length");
        long contentLength = StringUtils.isBlank(contentLengthStr) ? 0 : Long.valueOf(contentLengthStr);
        // 如果是文件上传 || 内容大小大于10M，那么不录制
        return !(contentType.contains(MediaType.MULTIPART_FORM_DATA_VALUE) || contentLength > MAX_CONTENT_LENGTH);
    }

    /**
     * 请求是否是feign调用
     * @param request
     * @return
     */
    private boolean isFeignRequest(HttpServletRequest request) {
        return "true".equals(request.getHeader(CustomRequestInterceptor.X_RPC));
    }

    @Override
    public void destroy() {
        // do nothing
    }
}
