package cn.com.duibaboot.ext.autoconfigure.grouping.httpclient;

import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;

import java.io.IOException;
import java.util.concurrent.Future;

/**
 * CloseableHttpAsyncClient的包装类，让CloseableHttpAsyncClient支持多场景测试的服务分组
 */
public class ServiceGroupHttpAsyncClientWrapper extends CloseableHttpAsyncClient {

    private CloseableHttpAsyncClient original;

    public ServiceGroupHttpAsyncClientWrapper(CloseableHttpAsyncClient original){
        this.original = original;
    }

    private <T> FutureCallback<T> getServiceGroupWrappedCallback(FutureCallback<T> callback){
        final String currentGroupKey = ServiceGroupContext.getCurrentGroupKey();
        FutureCallback<T> wrappedCallback = new FutureCallback<T>() {
            @Override
            public void completed(T result) {
                ServiceGroupContext.setGroupKey(currentGroupKey);
                try {
                    callback.completed(result);
                } finally {
                    ServiceGroupContext.removeGroupKey();
                }
            }

            @Override
            public void failed(Exception ex) {
                ServiceGroupContext.setGroupKey(currentGroupKey);
                try {
                    callback.failed(ex);
                } finally {
                    ServiceGroupContext.removeGroupKey();
                }
            }

            @Override
            public void cancelled() {
                ServiceGroupContext.setGroupKey(currentGroupKey);
                try {
                    callback.cancelled();
                } finally {
                    ServiceGroupContext.removeGroupKey();
                }}
        };
        return wrappedCallback;
    }

    @Override
    public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer, HttpAsyncResponseConsumer<T> responseConsumer, HttpContext context, FutureCallback<T> callback) {
        return original.execute(requestProducer, responseConsumer, context, getServiceGroupWrappedCallback(callback));
    }

    @Override
    public boolean isRunning() {
        return original.isRunning();
    }

    @Override
    public void start() {
        original.start();
    }

    @Override
    public <T> Future<T> execute(HttpAsyncRequestProducer requestProducer, HttpAsyncResponseConsumer<T> responseConsumer, FutureCallback<T> callback) {
        return original.execute(requestProducer, responseConsumer, getServiceGroupWrappedCallback(callback));
    }

    @Override
    public Future<HttpResponse> execute(HttpHost target, HttpRequest request, HttpContext context, FutureCallback<HttpResponse> callback) {
        return original.execute(target, request, context, getServiceGroupWrappedCallback(callback));
    }

    @Override
    public Future<HttpResponse> execute(HttpHost target, HttpRequest request, FutureCallback<HttpResponse> callback) {
        return original.execute(target, request, getServiceGroupWrappedCallback(callback));
    }

    @Override
    public Future<HttpResponse> execute(HttpUriRequest request, HttpContext context, FutureCallback<HttpResponse> callback) {
        return original.execute(request, context, getServiceGroupWrappedCallback(callback));
    }

    @Override
    public Future<HttpResponse> execute(HttpUriRequest request, FutureCallback<HttpResponse> callback) {
        return original.execute(request, getServiceGroupWrappedCallback(callback));
    }

    @Override
    public void close() throws IOException {
        original.close();
    }
}
