package cn.com.duibaboot.ext.autoconfigure.limiter;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.boot.utils.AopTargetUtils;
import com.google.common.collect.Sets;
import feign.Feign;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.config.annotation.Service;
import org.apache.dubbo.rpc.model.ApplicationModel;
import org.apache.dubbo.rpc.model.ProviderMethodModel;
import org.apache.dubbo.rpc.model.ProviderModel;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.EventListener;
import org.springframework.web.bind.annotation.RestController;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-12-30 16:01
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@Import({ServerLimiterAutoConfiguration.class, SpringMvcConfigForLimiter.class})
public class ServerApiAutoConfiguration {

    private static final Set<String> selfFeignClients = Sets.newHashSet();

    public static Set<String> getSelfFeignClients() {
        return selfFeignClients;
    }

    @Configuration
    @ConditionalOnClass({Service.class, ApplicationModel.class})
    public static class dubboServerApiAutoConfiguration {

        @EventListener(MainContextRefreshedEvent.class)
        public void initSelfFeignClients() {

            Collection<ProviderModel> providerModels = ApplicationModel.allProviderModels();
            if(providerModels.isEmpty()) {
                return;
            }

            providerModels.forEach(providerModel -> {

                List<ProviderMethodModel> providerMethodModels = providerModel.getAllMethods();
                Class<?> interfaceClass = providerModel.getServiceInterfaceClass();
                providerMethodModels.forEach(providerMethodModel -> selfFeignClients.add(Feign.configKey(interfaceClass, providerMethodModel.getMethod())));
            });
        }
    }

    @Configuration
    @ConditionalOnClass({AdvancedFeignClient.class, RestController.class})
    public static class springCloudServerApiAutoConfiguration implements ApplicationContextAware {

        @Override
        public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {

            Map<String, Object> beanMap = applicationContext.getBeansWithAnnotation(RestController.class);
            buildBeanDefinition(beanMap, (Class<?> classInstance) -> classInstance.isAnnotationPresent(AdvancedFeignClient.class));
        }

        private void buildBeanDefinition(Map<String, Object> beanMap, Function<Class<?>, Boolean> isAnnotation) {

            if (beanMap == null || beanMap.size() == 0) {
                return;
            }

            beanMap.values().forEach(bean -> {

                Object aopTargetObject = bean;
                try {
                    while (AopTargetUtils.getTarget(aopTargetObject) != aopTargetObject) {
                        aopTargetObject = AopTargetUtils.getTarget(aopTargetObject);
                    }
                } catch (Exception e) {
                    aopTargetObject = bean;
                }

                Class<?>[] interfaceClasses = aopTargetObject.getClass().getInterfaces();
                if (interfaceClasses == null || interfaceClasses.length == 0) {
                    return;
                }

                buildUniqKey(interfaceClasses, isAnnotation);

            });

        }

        private void buildUniqKey(Class<?>[] interfaceClasses, Function<Class<?>, Boolean> isAnnotation) {

            for (Class<?> interfaceClass : interfaceClasses) {

                try {

                    if (!isAnnotation.apply(interfaceClass)) {
                        continue;
                    }

                    Method[] methods = interfaceClass.getMethods();
                    for (Method method : methods) {
                        selfFeignClients.add(Feign.configKey(interfaceClass, method));
                    }

                } catch (Exception e) {
                    log.info("加载springCloud接口错误, class:{}", interfaceClass, e);
                }
            }
        }
    }



    @Configuration
    public static class ServerApiEndpointConfiguration {
        @Bean
        public ServerApiEndpoint serverApiEndpointConfiguration() {
            return new ServerApiEndpoint();
        }
    }

}
