package cn.com.duibaboot.ext.autoconfigure.limiter;

import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.boot.utils.RequestUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.CustomRequestInterceptor;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import com.google.common.util.concurrent.RateLimiter;
import com.netflix.concurrency.limits.Limiter;
import feign.Feign;
import lombok.extern.slf4j.Slf4j;
import org.apache.dubbo.rpc.Result;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.web.BasicErrorController;
import org.springframework.cloud.netflix.feign.FeignClient;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.Servlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2020-01-02 14:41
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@ConditionalOnClass({Servlet.class, FeignClient.class})
@ConditionalOnWebApplication
public class SpringRpcLimterInterceptor implements HandlerInterceptor {

    private static final ThreadLocal<Limiter.Listener> localLimiter = new ThreadLocal<>();

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        if (!(handler instanceof HandlerMethod) || (RequestUtils.getRequestPath(request).equals("/error")
                && request.getAttribute("javax.servlet.forward.request_uri") != null)) {
            return true;
        }

        //只处理RPC请求（来自FeignClient的rest rpc调用）,如果请求头中含有特殊字段，才认为是rpc请求
        if (!"true".equals(request.getHeader(CustomRequestInterceptor.X_RPC))) {
            return true;
        }

        HandlerMethod handlerMethod = (HandlerMethod) handler;
        Method method = handlerMethod.getMethod();
        Class<?> handlerClass = method.getDeclaringClass();
        if (handlerClass.equals(BasicErrorController.class)) {
            return true;
        }
        Class<?> feignClientClass = null;
        //找到接口中注解了@FeignClient或@AdvancedFeignClient的接口
        for (Class<?> clazz : handlerClass.getInterfaces()) {
            if (clazz.isAnnotationPresent(AdvancedFeignClient.class) || clazz.isAnnotationPresent(FeignClient.class)) {
                feignClientClass = clazz;
                break;
            }
        }

        if (feignClientClass == null) {
            log.warn("[NOTIFYME]can not find any interface of class:[{}] which annotated with @FeignClient or @AdvancedFeignClient, will not limit", handlerClass.getName());
            return true;
        }

        method = ReflectionUtils.findMethod(feignClientClass, method.getName(), method.getParameterTypes());

        if (method == null) {
            log.warn("[NOTIFYME]feign method is null");
            return true;
        }

        String feignUniqKey = Feign.configKey(feignClientClass, method);
        Object limiter = ServerLimiterAutoConfiguration.getServerLimitRule(feignUniqKey);
        if (limiter == null) {
            return true;
        }

        if (limiter instanceof RateLimiter) {
            RateLimiter rateLimiter = (RateLimiter) limiter;
            if (!rateLimiter.tryAcquire()) {
                throw new RateLimiterException();
            }

            return true;
        }

        Limiter<Void> vegasLimiter = (Limiter) limiter;
        Limiter.Listener listener = vegasLimiter.acquire(null).orElseThrow(RateLimiterException::new);
        localLimiter.set(listener);
        return true;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {
        // ignore
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {

        Limiter.Listener listener = localLimiter.get();
        if(listener == null) {
            return;
        }

        if(ex == null) {
            listener.onSuccess();
        }else {
            listener.onIgnore();
        }

        localLimiter.remove();
    }
}
