package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.hystrix;

import com.netflix.hystrix.HystrixCommand;
import com.netflix.hystrix.HystrixInvokable;
import com.netflix.hystrix.strategy.executionhook.HystrixCommandExecutionHook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * hyxtrix插件，使用spi机制生效<br/>
 * spring-cloud feign调用如果设置了降级，则降级前触发的异常不会打到日志里，从而不能及时发现线上问题。这个类把错误信息强制打印到日志里
 */
public class DuibaHystrixCommandExecutionHook extends HystrixCommandExecutionHook {

    private static final Logger logger = LoggerFactory.getLogger(DuibaHystrixCommandExecutionHook.class);

    @Override
    public <T> Exception onFallbackError(HystrixInvokable<T> commandInstance, Exception e) {
        return super.onFallbackError(commandInstance, e);
    }

    // 如果execute出错了并且没有实现自定义降级逻辑，会进入onFallbackError回调（因为默认的实现会抛出UnsupportedOperationException），而不会进入onFallbackSuccess，
    // 如果实现了自定义降级逻辑，并且降级代码没有出错，则会进入onFallbackSuccess回调，在这里记录异常非常合适
    @Override
    public <T> void onFallbackSuccess(HystrixInvokable<T> commandInstance) {
        if(commandInstance instanceof HystrixCommand){
            HystrixCommand c = (HystrixCommand)commandInstance;
            Throwable e = c.getExecutionException();
            if(e!=null) {
                logger.warn("hystrix group:{}, key:{} execute failed and entered fallback", c.getCommandGroup().name(), c.getCommandKey().name(), e);
            }
        }
    }
}
