package cn.com.duibaboot.ext.autoconfigure.devtools;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.devtools.classpath.ClassPathFileSystemWatcher;
import org.springframework.boot.devtools.classpath.PatternClassPathRestartStrategy;
import org.springframework.boot.devtools.filewatch.ChangedFile;
import org.springframework.boot.devtools.restart.ConditionalOnInitializedRestarter;
import org.springframework.boot.devtools.restart.classloader.RestartClassLoader;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.ReflectionUtils;

import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

//TODO 热重启时可能需要清理某些缓存，清理缓存可参考 spring中 Restarter.clear 方法的写法，监听ApplicationStartingEvent来清理
/**
 * DevTools拦截器, 配合BootReload chrome插件使用，提高开发效率
 */
@Configuration
@ConditionalOnWebApplication
public class DevToolsAutoConfiguration {

    @ConditionalOnClass(RestartClassLoader.class)
    @ConditionalOnInitializedRestarter
    @ConditionalOnProperty(prefix = "spring.devtools.restart", name = "enabled", matchIfMissing = true)
    static class DuibaDevToolsRestartConfiguration {

        /**
         * 替换PatternClassPathRestartStrategy策略后，从而让spring-boot-devtools遇到任意class文件更新都不会自动热重启，热重启改为在BootDevToolsFilter中手动触发
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor classPathRestartStrategyPostProcessor(){
            return new SpecifiedBeanPostProcessor<ClassPathFileSystemWatcher>(){

                @Override
                public int getOrder() {
                    return 0;
                }

                @Override
                public Class<ClassPathFileSystemWatcher> getBeanType() {
                    return ClassPathFileSystemWatcher.class;
                }

                @Override
                public Object postProcessBeforeInitialization(ClassPathFileSystemWatcher bean, String beanName) throws BeansException {
                    Field field = ReflectionUtils.findField(bean.getClass(), "restartStrategy");
                    field.setAccessible(true);

                    String[] excludes = new String[]{"/**"};
                    PatternClassPathRestartStrategy s = new PatternClassPathRestartStrategy(excludes){
                        @Override
                        public boolean isRestartRequired(ChangedFile file) {
                            return false;
                        }
                    };

                    ReflectionUtils.setField(field, bean, s);
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(ClassPathFileSystemWatcher bean, String beanName) throws BeansException {
                    return bean;
                }
            };
        }

        @Bean
        public ClassPathChangedFilesContainer classPathChangedFilesContainer(ApplicationContext applicationContext) {
            return new ClassPathChangedFilesContainer(applicationContext);
        }

    }

    /**
     * DevTools拦截器, 配合BootReload chrome插件使用，提高开发效率,for servlet
     */
    @Configuration
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    public static class BootDevToolsServletFilterConfiguration{
        @Resource
        private ApplicationContext applicationContext;

        @Autowired(required = false)
        private ClassPathChangedFilesContainer classPathChangedFilesContainer;

        @Bean
        public FilterRegistrationBean bootDevToolsFilterConfigurer(){
            BootDevToolsFilter filter=new BootDevToolsFilter(applicationContext, classPathChangedFilesContainer);
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<>();
            urlPatterns.add("/devtools/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 1);//次高优先级
            return registrationBean;
        }
    }

    /**
     * DevTools拦截器, 配合BootReload chrome插件使用，提高开发效率,for webflux/reactive
     */
    @Configuration
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.REACTIVE)
    public static class BootDevToolsReactiveFilterConfiguration{
        @Resource
        private ApplicationContext applicationContext;

        @Autowired(required = false)
        private ClassPathChangedFilesContainer classPathChangedFilesContainer;

        @Bean
        @Order(Ordered.HIGHEST_PRECEDENCE + 1)//次高优先级
        public BootDevToolsReactiveFilter bootDevToolsReactiveFilter(){
            return new BootDevToolsReactiveFilter(applicationContext, classPathChangedFilesContainer);
        }
    }

}
